import {FC, memo, useCallback} from 'react';

import {IWithClassName} from 'types/withClassName';

import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {deviceModMobile} from 'utilities/stylesUtils';
import {CurrencyType} from 'utilities/currency/CurrencyType';
import useImmutableCallback from 'utilities/hooks/useImmutableCallback';
import {getChangedValuesInRange} from 'projects/hotels/components/HotelsFilters/components/PriceFilter/utilities/rangeSteps';

import InputRange, {IInputRangeProps} from 'components/InputRange/InputRange';
import Price from 'components/Price/Price';
import Text from 'components/Text/Text';

import cx from './Range.scss';

export interface IHotelsPriceFilterRangeProps extends IWithClassName {
    max: number;
    min: number;
    value: [number, number];
    enableTooltip?: boolean;
    disabled?: boolean;
    currency: CurrencyType;
    onChange(value: [number, number]): void;
    onUpdate?(value: [number, number]): void;
}

const HotelsPriceFilterRange: FC<IHotelsPriceFilterRangeProps> = ({
    min,
    max,
    value,
    enableTooltip = false,
    currency,
    onChange,
    onUpdate,
    className,
    ...props
}: IHotelsPriceFilterRangeProps) => {
    const deviceType = useDeviceType();
    const renderHandleTooltip = useCallback(
        ({
            value,
            alignment,
        }: Parameters<
            Required<IInputRangeProps>['renderHandleTooltip']
        >[0]) => (
            <Text size="s">
                <Price
                    value={value}
                    isFrom={alignment === 'left'}
                    isUpTo={alignment === 'right'}
                    postfix={value === max ? '+' : ''}
                    currency={currency}
                />
            </Text>
        ),
        [currency, max],
    );

    const handleChange = useImmutableCallback((next: [number, number]) => {
        onChange(getChangedValuesInRange(next, value, max - min));
    });
    const handleUpdate = useImmutableCallback((next: [number, number]) => {
        onUpdate?.(getChangedValuesInRange(next, value, max - min));
    });

    return (
        <div
            className={cx(
                'root',
                deviceModMobile('root', deviceType),
                className,
            )}
        >
            <InputRange
                rangeValues={[min, max]}
                values={value}
                onChange={handleChange as IInputRangeProps['onChange']}
                onUpdate={handleUpdate as IInputRangeProps['onUpdate']}
                tooltipPosition="top"
                canRenderTooltip={enableTooltip}
                renderHandleTooltip={renderHandleTooltip}
                size={deviceType.isMobile ? 'm' : 's'}
                {...props}
            />
        </div>
    );
};

export default memo(HotelsPriceFilterRange);
