import {ForwardedRef, forwardRef} from 'react';

import {IWithClassName} from 'types/withClassName';

import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import useImmutableCallback from 'utilities/hooks/useImmutableCallback';
import {CurrencyType} from 'utilities/currency/CurrencyType';
import {useDeviceType} from 'utilities/hooks/useDeviceType';

import * as i18nBlock from 'i18n/hotels-HotelsPriceFilterWithInputs';

import PriceInput from 'components/PriceInput/PriceInput';
import Flex from 'components/Flex/Flex';
import {TBoxSizes} from 'components/Box/Box';

export interface IHotelsPriceFilterRangeInputsProps
    extends IWithQaAttributes,
        IWithClassName {
    max: number;
    min: number;
    value: [number, number];
    currency: CurrencyType;
    disabled?: boolean;
    onChange(value: [number, number]): void;

    // TODO Удалить после синхронизации отступов
    spaceBetween?: TBoxSizes;
}

const HotelsPriceFilterRangeInputs = (
    {
        min,
        max,
        value,
        disabled,
        currency,
        onChange,
        className,
        spaceBetween,
        ...props
    }: IHotelsPriceFilterRangeInputsProps,
    ref: ForwardedRef<HTMLDivElement>,
) => {
    const deviceType = useDeviceType();
    const handleFromChange = useImmutableCallback((changedValue: number) =>
        onChange([Math.min(Math.max(changedValue, min), value[1]), value[1]]),
    );
    const handleToChange = useImmutableCallback((changedValue: number) =>
        onChange([value[0], Math.max(Math.min(changedValue, max), value[0])]),
    );

    return (
        <Flex
            flexDirection="row"
            flexWrap="nowrap"
            inline
            ref={ref}
            // TODO Узнать - почему разные?
            between={spaceBetween ?? (deviceType.isMobile ? 3 : 2)}
            className={className}
        >
            <PriceInput
                min={min}
                max={Math.min(max, value[1])}
                value={value[0]}
                disabled={disabled}
                currency={currency}
                // TODO Возможно, стоит заменить поведение onChange/добавить свойство, регулирующее поведение
                onValueUpdate={handleFromChange}
                onValueChange={handleFromChange}
                beforeText={i18nBlock._from()}
                {...prepareQaAttributes({
                    ...props,
                    current: 'from',
                })}
            />
            <PriceInput
                min={Math.max(min, value[0])}
                max={max}
                value={value[1]}
                disabled={disabled}
                currency={currency}
                onValueUpdate={handleToChange}
                onValueChange={handleToChange}
                postfix={value[1] === max ? '+' : ''}
                beforeText={i18nBlock.to()}
                {...prepareQaAttributes({
                    ...props,
                    current: 'to',
                })}
            />
        </Flex>
    );
};

export default forwardRef(HotelsPriceFilterRangeInputs);
