import React, {useEffect, useCallback, useRef} from 'react';
import {useSelector} from 'react-redux';
import _times from 'lodash/times';

import {ADFOX_BANNER_SEARCH_POSITION} from 'constants/AdFox';
import {SUBSCRIPTION_BLOCK_OFFSET} from 'constants/subscription';

import {IWithDeviceType} from 'types/withDeviceType';
import {PermalinkType} from 'types/hotels/hotel/IHotel';
import {IRequiredOfferParams} from 'types/hotels/offer/IHotelOffer';
import {THotelWithOffersByPermalink} from 'types/hotels/hotel/IHotelWithOffers';
import {IActiveHotel} from 'types/hotels/hotel/IActiveHotel';
import {EHotelsGoal} from 'utilities/metrika/types/goals/hotels';
import {IWithClassName} from 'types/withClassName';
import {ESubscriptionVerticalName} from 'types/subscription/ESubscriptionVerticalName';
import {ESubscriptionSource} from 'types/subscription/ESubscriptionSource';

import {ISearchHotelsData} from 'reducers/hotels/searchPage/search/reducer';

import {getSearchInfo} from 'selectors/hotels/search/searchHotels/getSearchInfo';
import experimentsSelector from 'selectors/common/experimentsSelector';
import {hotelsCleanSerpEnabledSelector} from 'selectors/hotels/search/hotelsCleanSerpEnabledSelector';

import {params, reachGoal} from 'utilities/metrika';
import {deviceModMobile} from 'utilities/stylesUtils';
import {atPlaceOrEarly} from 'utilities/actions/atPlaceOrEarly';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import getColorFromRequestsTime from 'projects/hotels/utilities/getColorFromRequestsTime/getColorFromRequestsTime';

import OptionalSubscription from 'containers/Subscriptions/Inline/OptionalSubscription';

import NoMoreFavoritesModal from 'projects/favorites/components/NoMoreFavoritesModal/NoMoreFavoritesModal';
import HotelCard from 'projects/hotels/components/HotelCard/HotelCard';
import HotelSearchCardSkeleton from 'projects/hotels/pages/SearchPage/components/HotelSearchCardSkeleton/HotelSearchCardSkeleton';
import MobileHotelSearchCardSkeleton from 'projects/hotels/pages/SearchPage/components/MobileHotelSearchCardSkeleton/MobileHotelSearchCardSkeleton';
import FilterResets from 'projects/hotels/pages/SearchPage/components/BaseSearchPage/components/FilterResets/FilterResets';

import ScopeContext from 'contexts/ScopeContext';

import Snackbar from '../Snackbar/Snackbar';

import cx from './HotelsList.scss';

export enum EHotelsListSkeletonType {
    CARD_SKELETON = 'cardSkeleton',
    OFFER_SKELETON = 'offerSkeleton',
}

interface ISearchInfoFromProps {
    isLoading: boolean;
    isError: boolean;
    lastSearchTimestamp: number | null;
    data?: ISearchHotelsData;
    startSearchTime?: number;
    noMoreFavoritesModalIsOpen?: boolean;
}

export interface IFavoriteClick {
    permalink: PermalinkType;
    isFavorite: boolean;
}

/* Component Types */
interface IHotelsList
    extends IWithClassName,
        IWithDeviceType,
        IWithQaAttributes {
    hotels?: PermalinkType[];
    hotelWithOffersByPermalink?: THotelWithOffersByPermalink;
    nights: number;
    searchIsFinished?: boolean;
    offerRequestParams?: IRequiredOfferParams;
    activeHotelPermalink?: PermalinkType;
    onHoverHotelCard: (permalink: IActiveHotel | undefined) => void;
    onLeaveHotelCard: () => void;
    searchInfoFromProps?: ISearchInfoFromProps;
    totalHotelsList?: number;
    skeletonType?: EHotelsListSkeletonType;
    onFavoriteClick?: (params: IFavoriteClick) => void;
    onCloseModalClick?: () => void;
    snackbarClassName?: string;
    withTopHotel?: boolean;
    currentPage?: 'favorites' | 'search';
    hideNoFavoritesHotels?: boolean;
}

/* Constants */
const TOTAL_HOTEL_LIST = 10;

const HotelsList: React.FC<IHotelsList> = props => {
    const {
        hotels = [],
        className,
        nights,
        offerRequestParams,
        searchIsFinished,
        onHoverHotelCard,
        onLeaveHotelCard,
        activeHotelPermalink,
        hotelWithOffersByPermalink = {},
        withTopHotel,
        deviceType,
        searchInfoFromProps,
        totalHotelsList,
        onFavoriteClick,
        skeletonType = EHotelsListSkeletonType.CARD_SKELETON,
        onCloseModalClick,
        snackbarClassName,
        currentPage,
        hideNoFavoritesHotels = false,
    } = props;
    const {isMobile} = deviceType;

    const rootRef = useRef<HTMLDivElement>(null);
    const searchInfoFromState = useSelector(getSearchInfo);
    const searchInfo = searchInfoFromProps || searchInfoFromState;
    const startSearchTime = searchInfo.startSearchTime;

    const experiments = useSelector(experimentsSelector);
    const noMoreFavoritesModalIsVisible = searchInfo.noMoreFavoritesModalIsOpen;
    const {passToMetricHotelsSearchPageSearchtime} = experiments;
    const hotelsCleanSerpEnabled = useSelector(hotelsCleanSerpEnabledSelector);

    const handleCardClick = useCallback(
        (searchedByUser: boolean): void => {
            if (withTopHotel) {
                reachGoal(
                    searchedByUser
                        ? EHotelsGoal.SEARCH_PAGE_FIRST_SNIPPET_CLICK
                        : EHotelsGoal.SEARCH_PAGE_OTHER_SNIPPET_CLICK,
                );
            }
        },
        [withTopHotel],
    );

    const withInfiniteScroll = isMobile;
    const hotelList =
        searchIsFinished || withInfiniteScroll
            ? hotels
            : _times(
                  totalHotelsList || TOTAL_HOTEL_LIST,
                  index => hotels && hotels[index],
              );
    const canRenderResetList = currentPage === 'search' && searchIsFinished;

    function getHotelCard(
        permalink: PermalinkType,
        index: number,
    ): React.ReactElement {
        const hotelWithOffers = hotelWithOffersByPermalink[permalink];

        if (hotelWithOffers && permalink) {
            const isActive = permalink === activeHotelPermalink;

            return (
                <HotelCard
                    className={cx(
                        'hotelCard',
                        deviceModMobile('hotelCard', deviceType),
                        {
                            hotelCard_active: isActive,
                        },
                    )}
                    key={permalink}
                    nights={nights}
                    currentPage={currentPage}
                    offerRequestParams={offerRequestParams}
                    onHoverHotel={onHoverHotelCard}
                    onLeaveHotel={onLeaveHotelCard}
                    onCardClick={handleCardClick}
                    hotelWithOffers={hotelWithOffers}
                    onFavoriteClick={onFavoriteClick}
                    skeletonType={skeletonType}
                    renderOfferSkeleton={
                        (searchInfo.isLoading || !searchIsFinished) &&
                        skeletonType === EHotelsListSkeletonType.OFFER_SKELETON
                    }
                    isActive={isActive}
                    isLazyImagesCarousel={isMobile ? index > 1 : index > 4}
                    {...prepareQaAttributes({
                        parent: props,
                        current: 'hotelCard',
                        key: permalink,
                    })}
                />
            );
        }

        if (isMobile) {
            return (
                <MobileHotelSearchCardSkeleton
                    key={index}
                    className={cx('hotelCard', 'hotelCard_mobile')}
                />
            );
        }

        return (
            <HotelSearchCardSkeleton key={index} className={cx('hotelCard')} />
        );
    }

    useEffect(() => {
        if (
            passToMetricHotelsSearchPageSearchtime &&
            searchIsFinished &&
            startSearchTime
        ) {
            const endRenderTime = Date.now();
            const zone = getColorFromRequestsTime(
                endRenderTime - startSearchTime,
            );

            params({
                hotels: {
                    searchPage: {searchtime: {[zone]: 1}},
                },
            });
        }
        // startSearchTime нет в зависимостях, так как экшн clearSearchBeforeNewSearch
        // вызывается лишний раз и еще раз меняет startSearchTime
        // Нужно будет убрать этот хак, когда пофиксится глобальная проблема с повторным вызовом прошлых экшнов
    }, [searchIsFinished, passToMetricHotelsSearchPageSearchtime]);

    return (
        <ScopeContext.Provider
            value={rootRef.current ? rootRef.current : undefined}
        >
            <div
                className={cx('root', className)}
                ref={rootRef}
                {...prepareQaAttributes({
                    parent: props,
                    current: searchIsFinished
                        ? 'searchIsFinished'
                        : 'searchIsPending',
                })}
            >
                {canRenderResetList && <FilterResets />}
                {hotelList.map((permalink: PermalinkType, index) => {
                    if (
                        hotelWithOffersByPermalink[permalink] &&
                        !hotelWithOffersByPermalink[permalink].hotel
                            .isFavorite &&
                        hideNoFavoritesHotels
                    ) {
                        return null;
                    }

                    // insert subscription block
                    if (
                        !hotelsCleanSerpEnabled &&
                        atPlaceOrEarly(
                            hotelList.length,
                            index,
                            ADFOX_BANNER_SEARCH_POSITION +
                                SUBSCRIPTION_BLOCK_OFFSET,
                        )
                    ) {
                        return (
                            <React.Fragment key={permalink}>
                                {getHotelCard(permalink, index)}
                                <OptionalSubscription
                                    className={cx(
                                        'subscription',
                                        deviceModMobile(
                                            'subscription',
                                            deviceType,
                                        ),
                                    )}
                                    key="subscription"
                                    vertical={ESubscriptionVerticalName.Hotels}
                                    source={ESubscriptionSource.SEARCH}
                                    type="thin"
                                />
                            </React.Fragment>
                        );
                    }

                    return getHotelCard(permalink, index);
                })}
                <Snackbar
                    deviceType={deviceType}
                    className={snackbarClassName}
                    currentPage={currentPage || null}
                />
                <NoMoreFavoritesModal
                    onClose={onCloseModalClick}
                    isVisible={Boolean(noMoreFavoritesModalIsVisible)}
                />
            </div>
        </ScopeContext.Provider>
    );
};

export default React.memo(HotelsList);
