import {FC, memo, useCallback, useMemo} from 'react';
import {useDispatch, useSelector} from 'react-redux';

import {EHotelsGoal} from 'utilities/metrika/types/goals/hotels';
import {ISortTypeGroup, TSortId} from 'types/hotels/search/ISortInfo';

import {openHotelsGeolocationErrorModalAction} from 'reducers/common/hotelsGeolocationErrorModal/actions';
import {setActiveSort as setActiveSortAction} from 'reducers/hotels/searchPage/sort/actions';

import {hotelsSortSelector} from 'projects/hotels/components/HotelsSort/selectors';

import {reachGoal} from 'utilities/metrika';
import browserGeolocation from 'utilities/browserGeolocation/browserGeolocation';
import type {IWithQaAttributes} from 'utilities/qaAttributes/qaAttributes';
import useDispatchedAction from 'utilities/hooks/useDispatchedAction';
import useImmutableCallback from 'utilities/hooks/useImmutableCallback';

import HotelsSortView, {
    IHotelsSortView,
} from 'projects/hotels/components/HotelsSort/components/View';
import {TButtonTheme} from 'components/Button/Button';

export interface IHotelsSortProps
    extends IWithQaAttributes,
        Pick<IHotelsSortView, 'className' | 'showIcon' | 'menuWidth' | 'size'> {
    theme?: TButtonTheme;
}

type TGroupById = Record<TSortId, ISortTypeGroup[]>;

const groupByIdReducer = (
    acc: TGroupById,
    group: ISortTypeGroup,
): TGroupById => {
    for (const {id} of group.sortTypes) {
        acc[id] ||= [];
        acc[id].push(group);
    }

    return acc;
};

const HotelsSort: FC<IHotelsSortProps> = props => {
    const {needSyncSortWithServer, sortInfo} = useSelector(hotelsSortSelector);

    const dispatch = useDispatch();
    const openGeoModalError = useDispatchedAction(
        openHotelsGeolocationErrorModalAction,
        dispatch,
    );
    const setActiveSort = useDispatchedAction(setActiveSortAction, dispatch);

    const value = sortInfo?.selectedSortId;

    const {sortTypes, groupById} = useMemo(
        () => ({
            sortTypes:
                sortInfo?.availableSortTypeGroups.flatMap(
                    group => group.sortTypes,
                ) ?? [],
            groupById:
                sortInfo?.availableSortTypeGroups.reduce(
                    groupByIdReducer,
                    {},
                ) ?? {},
        }),
        [sortInfo],
    );

    const applyChangedValue = useCallback(
        (...params: Parameters<typeof setActiveSort>) => {
            setActiveSort(...params);
            reachGoal(EHotelsGoal.SEARCH_PAGE_SORT_CHANGE);
        },
        [setActiveSort],
    );
    const assertGeoAndApplyChangedValue = useCallback(
        async (id: TSortId) => {
            const {sortOrigin} = await browserGeolocation.getGeolocation();

            if (sortOrigin) {
                applyChangedValue({id, sortOrigin});
            } else {
                openGeoModalError();
            }
        },
        [openGeoModalError, applyChangedValue],
    );

    const onChange = useImmutableCallback((id: TSortId) => {
        if (!sortInfo || id === value) return;

        const {requiresGeoLocation} = groupById[id]?.[0] ?? {};

        if (requiresGeoLocation) {
            assertGeoAndApplyChangedValue(id);
        } else {
            applyChangedValue({id});
        }
    });

    return (
        <HotelsSortView
            {...props}
            types={sortTypes}
            value={value}
            onChange={onChange}
            skeleton={!sortInfo || !sortInfo.availableSortTypeGroups.length}
            loading={needSyncSortWithServer}
        />
    );
};

export default memo(HotelsSort);
