import {ChangeEvent, FC, useMemo} from 'react';

import {ISortViewProps} from 'projects/hotels/components/HotelsSort/types/SortViewProps';

import {useBoolean} from 'utilities/hooks/useBoolean';
import {prepareQaAttributes} from 'utilities/qaAttributes/qaAttributes';
import useImmutableCallback from 'utilities/hooks/useImmutableCallback';

import * as i18nCommonSort from 'i18n/common-sort';

import BottomSheet from 'components/BottomSheet/BottomSheet';
import {IButtonProps} from 'components/Button/Button';
import Radiobox from 'components/Radiobox/Radiobox';
import Text from 'components/Text/Text';
import Skeleton from 'components/Skeletons/Skeleton/Skeleton';
import ArrowBottom from 'icons/16/ArrowBottom';
import CheckButton from 'components/CheckButton/CheckButton';

import cx from '../HotelsSort.scss';

export interface IHotelsSortBottomSheetViewProps
    extends ISortViewProps,
        Pick<IButtonProps, 'size'> {}

const HotelsSortBottomSheetView: FC<IHotelsSortBottomSheetViewProps> = ({
    className,
    types,
    loading,
    value,
    onChange,
    skeleton,
    size = 'm',
    ...props
}) => {
    const visible = useBoolean(false);

    const options = useMemo(
        () =>
            types.map(({id, name}) => ({
                label: name,
                value: id,
                className: cx('item'),
            })),
        [types],
    );
    const selectedOption = useMemo(
        () => options.find(option => option.value === value),
        [options, value],
    );

    const handleChange = useImmutableCallback(
        (e: ChangeEvent<HTMLInputElement>) => {
            visible.setFalse();
            onChange(e.target.value);
        },
    );

    if (skeleton) {
        return <Skeleton className={cx('triggerSize')} withAnimation />;
    }

    return (
        <div className={className} {...prepareQaAttributes(props)}>
            <CheckButton
                size={size}
                checked={false}
                theme="secondary"
                onClick={visible.setTrue}
                className={cx('triggerSize')}
                disabled={loading}
                iconRight={
                    <ArrowBottom
                        className={cx(
                            'arrow_icon',
                            visible.value && 'arrow_icon_open',
                        )}
                    />
                }
            >
                {selectedOption?.label}
            </CheckButton>
            <BottomSheet isOpened={visible.value} onClose={visible.setFalse}>
                <Text className={cx('title')} weight="bold" size="xl" tag="div">
                    {i18nCommonSort.title()}
                </Text>
                <Radiobox
                    className={cx('radio')}
                    value={value}
                    options={options}
                    onChange={handleChange}
                />
            </BottomSheet>
        </div>
    );
};

export default HotelsSortBottomSheetView;
