import {FC, memo, useCallback, useMemo} from 'react';

import {ISortType, TSortId} from 'types/hotels/search/ISortInfo';
import {ISortViewProps} from 'projects/hotels/components/HotelsSort/types/SortViewProps';

import {prepareQaAttributes} from 'utilities/qaAttributes/qaAttributes';
import getSortGroupIcon from 'projects/hotels/components/HotelsSort/utilities/getSortGroupIcon';
import {useDeviceType} from 'utilities/hooks/useDeviceType';

import Select, {ISelectOption, ISelectProps} from 'components/Select/Select';
import {TButtonTheme} from 'components/Button/Button';

import cx from '../HotelsSort.scss';

const MAX_HEIGHT = 300;
const getOptionSortName = (item: ISelectOption<string, ISortType>) =>
    item.data.name;

export interface IHotelsSortSelectViewProps
    extends ISortViewProps,
        Pick<ISelectProps<any, any>, 'menuWidth' | 'size'> {
    theme?: TButtonTheme;
}

const HotelsSortSelectView: FC<IHotelsSortSelectViewProps> = ({
    className,
    theme,
    showIcon,
    types,
    loading,
    value,
    onChange,
    skeleton,
    size = 'm',
    menuWidth = 'auto',
    ...props
}) => {
    const deviceType = useDeviceType();

    const options = useMemo(
        () =>
            types.map(sortType => ({
                value: sortType.id,
                data: sortType,
            })),
        [types],
    );

    const renderValue = useCallback(
        (item: ISelectOption<string, ISortType>) => {
            const Icon = getSortGroupIcon(item.data.hint, deviceType.isDesktop);

            return (
                <div className={cx('sortSelectOption')}>
                    {showIcon && Icon && (
                        <Icon
                            width="16"
                            height="16"
                            className={cx('sortSelectOptionIcon')}
                        />
                    )}
                    {item.data.name}
                </div>
            );
        },
        [showIcon, deviceType],
    );

    if (skeleton) {
        return <div className={cx('sortSelectSkeleton')} />;
    }

    return (
        <div className={className} {...prepareQaAttributes(props)}>
            <Select<TSortId, ISortType>
                size={size}
                theme={theme}
                width="max"
                menuClassName={cx('sortSelectMenu')}
                menuWidth={menuWidth}
                disabled={loading}
                adaptiveArrowIcon
                onChange={onChange}
                value={value}
                options={options}
                menuMaxHeight={MAX_HEIGHT}
                fixPopupInSpring
                renderItem={getOptionSortName}
                renderValue={renderValue}
            />
        </div>
    );
};

export default memo(HotelsSortSelectView);
