import moment from 'moment';
import React, {useMemo} from 'react';

import {IWithClassName} from 'types/withClassName';
import {EToggler} from 'types/common/togglers/EToggler';

import {getNow} from 'utilities/dateUtils';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {deviceMods} from 'utilities/stylesUtils';
import useCookieToggler from 'utilities/hooks/useCookieToggler';
import {insertJSXIntoKey} from 'utilities/tanker/insertJSXIntoKey';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import * as i18nBlock from 'i18n/mirCashback';

import FireIcon from 'icons/24/Fire';
import Text from 'components/Text/Text';
import Card from 'components/Card/Card';
import Link from 'components/Link/Link';
import Flex from 'components/Flex/Flex';
import Separator from 'components/Separator/Separator';
import RotatingArrowIcon from 'components/RotatingArrowIcon/RotatingArrowIcon';
import MirCashbackLandingLink from 'projects/hotels/components/MirCashbackLandingLink/MirCashbackLandingLink';
import BlueLeavesIcon from 'projects/hotels/components/MirCashbackBanner/icons/BlueLeavesIcon/BlueLeavesIcon';
import GreenLeavesIcon from 'projects/hotels/components/MirCashbackBanner/icons/GreenLeavesIcon/GreenLeavesIcon';
import MirPercentageIcon from 'projects/hotels/components/MirCashbackBanner/icons/MirPercentageIcon/MirPercentageIcon';
import MirCardIcon from 'icons/36/MirCard';
import ArrowBottomIcon from 'icons/24/ArrowBottom';

import rosturizmImageUrl from 'images/rosturizm.png';

import cx from './MirCashbackBanner.scss';

const END_DATE_MIR_CASHBACK = '2022-04-16';

interface IMirCashbackBannerProps extends IWithClassName, IWithQaAttributes {
    type:
        | EToggler.MIR_POSSIBLE_CASHBACK_SEARCH
        | EToggler.MIR_POSSIBLE_CASHBACK_HOTEL
        | EToggler.MIR_CASHBACK_HOTEL;
}

const MirCashbackBanner: React.FC<IMirCashbackBannerProps> = props => {
    const {className, type} = props;

    const deviceType = useDeviceType();

    const {value: opened, toggle} = useCookieToggler(type);

    const descriptionText = useMemo(() => {
        if (type === EToggler.MIR_POSSIBLE_CASHBACK_SEARCH) {
            return insertJSXIntoKey(
                i18nBlock.searchPossibleCashbackDescription,
            )({
                link: (
                    <MirCashbackLandingLink key="link" text={i18nBlock.how()} />
                ),
            });
        }

        if (type === EToggler.MIR_POSSIBLE_CASHBACK_HOTEL) {
            return insertJSXIntoKey(i18nBlock.hotelPossibleCashbackDescription)(
                {
                    link: (
                        <MirCashbackLandingLink
                            key="link"
                            text={i18nBlock.how()}
                        />
                    ),
                },
            );
        }

        return insertJSXIntoKey(i18nBlock.hotelCashbackDescription)({
            loyaltyLink: (
                <Link
                    key="loyaltyLink"
                    url="https://privetmir.ru/rostourism/register/"
                    target="_blank"
                    text={i18nBlock.loyaltyLinkText()}
                    theme="brand"
                />
            ),
            partnerLink: (
                <MirCashbackLandingLink
                    key="partnerLink"
                    text={i18nBlock.partnerLinkText()}
                />
            ),
            aboutLink: (
                <MirCashbackLandingLink
                    key="aboutLink"
                    text={i18nBlock.aboutLinkText()}
                />
            ),
        });
    }, [type]);

    const descriptionBlock = useMemo(() => {
        return (
            <Text className={cx('description')} color="current" tag="div">
                {descriptionText}
            </Text>
        );
    }, [descriptionText]);

    const dayLeftLabelText = useMemo(() => {
        const dayLeft = moment(END_DATE_MIR_CASHBACK).diff(
            moment(getNow()),
            'days',
            true,
        );

        if (dayLeft > 0 && dayLeft <= 1) {
            return i18nBlock.lastDayLabel();
        }

        if (dayLeft > 0 && dayLeft <= 14) {
            return i18nBlock.someDayLabel({dayLeft: Math.ceil(dayLeft)});
        }

        return null;
    }, []);

    return (
        <Card
            className={cx(
                'root',
                {
                    root_searchPossibleCashback:
                        type === EToggler.MIR_POSSIBLE_CASHBACK_SEARCH,
                    root_hotelPossibleCashback:
                        type === EToggler.MIR_POSSIBLE_CASHBACK_HOTEL,
                    root_hotelCashback: type === EToggler.MIR_CASHBACK_HOTEL,
                    root_hasBadge: dayLeftLabelText,
                },
                deviceMods('root', deviceType),
                className,
            )}
            radius="m"
            {...prepareQaAttributes({
                parent: props,
                current: 'mirCashbackBanner',
            })}
        >
            {dayLeftLabelText && (
                <div className={cx('dayLeftLabel')}>
                    <FireIcon className={cx('dayLeftLabelIcon')} />
                    {dayLeftLabelText}
                </div>
            )}
            {type === EToggler.MIR_CASHBACK_HOTEL ? (
                <BlueLeavesIcon className={cx('leaves')} />
            ) : (
                <GreenLeavesIcon className={cx('leaves')} />
            )}

            {deviceType.isMobile && opened && descriptionBlock}

            <Flex className={cx('mainContentWrapper')} alignItems="center">
                <MirPercentageIcon className={cx('percentage')} />

                {deviceType.isDesktop && descriptionBlock}

                <img
                    className={cx('rosturizmIcon')}
                    src={rosturizmImageUrl}
                    alt="Ростуризм"
                />

                <Separator className={cx('separator')} isVertical />

                <MirCardIcon className={cx('mirIcon')} />

                {deviceType.isMobile && (
                    <RotatingArrowIcon
                        className={cx('arrowIcon')}
                        Icon={ArrowBottomIcon}
                        rotated={opened}
                        onClick={toggle}
                    />
                )}
            </Flex>
        </Card>
    );
};

export default React.memo(MirCashbackBanner);
