import React, {useCallback} from 'react';

import {
    INavigationTokens,
    NavigationTokenType,
} from 'types/hotels/search/INavigationTokens';
import {IWithClassName} from 'types/withClassName';

import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import * as i18nBlock from 'i18n/hotels-SearchPage';

import ArrowLeftIcon from 'icons/16/ArrowLeft';
import ArrowRightIcon from 'icons/16/ArrowRight';
import Button from 'components/Button/Button';
import NavigationButtonsSkeleton from 'projects/hotels/components/NavigationButtons/components/NavigationButtonsSkeleton/NavigationButtonsSkeleton';

import cx from './NavigationButtons.scss';

interface INavigationButtons extends IWithClassName, IWithQaAttributes {
    isLoading: boolean;
    isEmptyList: boolean;
    navigationTokens?: INavigationTokens;

    onChangeNavigationToken(token: NavigationTokenType): void;
}

const ICON_LEFT = <ArrowLeftIcon />;
const ICON_RIGHT = <ArrowRightIcon />;

const NavigationButtons: React.FC<INavigationButtons> = props => {
    const {
        className,
        isLoading,
        isEmptyList,
        navigationTokens,
        onChangeNavigationToken,
    } = props;

    const handleClickPrevNavigationButton = useCallback(() => {
        requestAnimationFrame(() => {
            if (navigationTokens) {
                onChangeNavigationToken(navigationTokens.prevPage);
            }
        });
    }, [navigationTokens, onChangeNavigationToken]);

    const handleClickNextNavigationButton = useCallback(() => {
        requestAnimationFrame(() => {
            if (navigationTokens) {
                onChangeNavigationToken(navigationTokens.nextPage);
            }
        });
    }, [navigationTokens, onChangeNavigationToken]);

    if (isLoading) {
        return <NavigationButtonsSkeleton className={className} />;
    }

    if (isEmptyList || !navigationTokens) {
        return null;
    }

    const onlyOnePage =
        !navigationTokens.prevPage && !navigationTokens.nextPage;

    if (onlyOnePage) {
        return null;
    }

    return (
        <div className={className}>
            <Button
                className={cx('navigationButton')}
                onClick={handleClickPrevNavigationButton}
                theme="raised"
                size="l"
                iconLeft={ICON_LEFT}
                disabled={!navigationTokens.prevPage}
                {...prepareQaAttributes({
                    parent: props,
                    current: 'prevButton',
                })}
            >
                {i18nBlock.previousNavigationButton()}
            </Button>
            <Button
                className={cx('navigationButton', 'navigationButton_next')}
                onClick={handleClickNextNavigationButton}
                theme="raised"
                size="l"
                disabled={!navigationTokens.nextPage}
                iconRight={ICON_RIGHT}
                {...prepareQaAttributes({
                    parent: props,
                    current: 'nextButton',
                })}
            >
                {i18nBlock.nextNavigationButton()}
            </Button>
        </div>
    );
};

export default NavigationButtons;
