import {Component, ReactNode} from 'react';

import {IWithClassName} from 'src/types/withClassName';
import {IWithDeviceType} from 'src/types/withDeviceType';
import {
    IBookHotelInfo,
    IBookOfferBedsGroup,
    IBookOfferMeal,
    IBookOfferRoom,
    IBookPartnerHotelInfo,
    IBookSearchParams,
} from 'server/api/HotelsBookAPI/types/IBookOffer';
import {IBookUserSelect} from 'types/hotels/book/ICreateOrder';
import {ICancellationInfo} from 'types/hotels/offer/IHotelOfferCancellationInfo';
import {EPansionType} from 'server/api/HotelsBookAPI/types/ITestBookOfferToken';

import {deviceMods} from 'utilities/stylesUtils';
import {getImagesBySize} from 'projects/hotels/utilities/getImagesBySize/getImagesBySize';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {CHAR_NBSP} from 'utilities/strings/charCodes';

import * as i18nBlock from 'i18n/hotels-BookPayPage';

import HotelAddressAndRating from 'projects/hotels/components/HotelAddressAndRating/HotelAddressAndRating';
import BookHotelAmenities from 'projects/hotels/components/BookHotelAmenities/BookHotelAmenities';
import BookMealInfo from 'projects/hotels/components/BookMealInfo/BookMealInfo';
import BookSearchParams from 'projects/hotels/components/BookSearchParams/BookSearchParams';
import LinkButton from 'components/LinkButton/LinkButton';
import Image from 'components/TravelImage/TravelImage';
import BookRoomDescription from 'projects/hotels/components/BookRoomDescription/BookRoomDescription';
import BookSettlementInfo from 'projects/hotels/components/BookSettlementInfo/BookSettlementInfo';
import BookSelectBedsGroups from 'projects/hotels/components/BookSelectBedsGroups/BookSelectBedsGroups';
import BookPartnerHotelInfo from 'projects/hotels/components/BookPartnerHotelInfo/BookPartnerHotelInfo';
import Separator from 'components/Separator/Separator';
import {TwoColumnLayout} from 'components/Layouts/TwoColumnLayout/TwoColumnLayout';
import Text from 'components/Text/Text';
import HotelsCancellationInfo, {
    EFormatLabel,
} from 'projects/hotels/components/HotelsCancellationInfo/HotelsCancellationInfo';
import Rating from 'components/Rating/Rating';
import HotelAddress from 'projects/hotels/components/HotelAddress/HotelAddress';
import HotelStars from 'components/HotelStars/HotelStars';

import cx from './OfferHotelInfo.scss';

const MOBILE_IMAGE_SIZE = '350px';
const THUMBNAIL_IMAGE_SIZE = MOBILE_IMAGE_SIZE;
const HOTEL_NAME_SYMBOLS_COUNT_FOR_SIZE_L = 80;

interface IOfferHotelInfoState {
    partnerModalVisible: boolean;
}

export interface IOfferHotelInfoProps
    extends IWithClassName,
        IWithDeviceType,
        IWithQaAttributes {
    partnerId: string;
    searchParams: IBookSearchParams;
    hotelInfo: IBookHotelInfo;
    roomInfo: IBookOfferRoom;
    cancellationInfo?: ICancellationInfo;
    mealInfo: IBookOfferMeal;
    partnerHotelInfo: IBookPartnerHotelInfo;
    bookUserSelect: IBookUserSelect;
    bedsGroups: IBookOfferBedsGroup[];
    setBedsGroupId: (bedGroup: {
        bedsGroupId: any;
        bedsGroupIsSelectedByUser?: boolean;
    }) => void;
}

class OfferHotelInfo extends Component<
    IOfferHotelInfoProps,
    IOfferHotelInfoState
> {
    state: IOfferHotelInfoState = {
        partnerModalVisible: false,
    };

    private showPartnerModal = (): void => {
        this.setState({partnerModalVisible: true});
    };

    private hidePartnerModal = (): void => {
        this.setState({partnerModalVisible: false});
    };

    private renderPartnerModal(): ReactNode {
        const {
            partnerHotelInfo,
            deviceType: {isMobile},
        } = this.props;
        const {partnerModalVisible} = this.state;

        return (
            partnerModalVisible && (
                <BookPartnerHotelInfo
                    isMobile={isMobile}
                    isVisible={true}
                    onHideModal={this.hidePartnerModal}
                    partnerHotelInfo={partnerHotelInfo}
                />
            )
        );
    }

    private renderName(): ReactNode {
        const {
            hotelInfo: {name, stars, rating, address},
            deviceType,
        } = this.props;

        return (
            <TwoColumnLayout
                deviceType={deviceType}
                rightColumnOffset={3}
                rightColumnWidth={18}
                preserveColumnsOnMobile
            >
                <TwoColumnLayout.LeftColumn>
                    <div className={cx('section', 'section_size_xs')}>
                        <LinkButton
                            className={cx('hotelName', {
                                hotelName_size_smaller:
                                    name.length >
                                    HOTEL_NAME_SYMBOLS_COUNT_FOR_SIZE_L,
                            })}
                            theme="black"
                            onClick={this.showPartnerModal}
                            {...prepareQaAttributes({
                                parent: this.props,
                                current: 'hotel-name',
                            })}
                        >
                            {name}
                            {CHAR_NBSP}
                            <HotelStars stars={stars} size="12" />
                        </LinkButton>
                        {deviceType.isMobile && (
                            <Rating
                                size="m"
                                className={cx('rating')}
                                rating={rating}
                            />
                        )}
                    </div>
                    {deviceType.isMobile && (
                        <HotelAddress
                            className={cx('section', `section_size_m`)}
                            onClick={this.showPartnerModal}
                            type="link"
                            multiLine
                            text={address}
                        />
                    )}
                </TwoColumnLayout.LeftColumn>
                <TwoColumnLayout.RightColumn>
                    {deviceType.isMobile && (
                        <span className={cx('smallHotelImage')}>
                            {this.renderRoomImage()}
                        </span>
                    )}
                </TwoColumnLayout.RightColumn>
            </TwoColumnLayout>
        );
    }

    private renderPartnerNameAndLink(): ReactNode {
        const {
            hotelInfo: {partnerHotelName},
            deviceType: {isDesktop},
        } = this.props;

        return (
            <div
                className={cx(
                    'section',
                    {section_size_l: isDesktop},
                    'partner',
                    'partner_inline',
                )}
            >
                <p className={cx('partnerName')}>
                    {Boolean(partnerHotelName) &&
                        i18nBlock.partnerHotelName({
                            hotelName: partnerHotelName,
                        })}{' '}
                    <LinkButton
                        className={cx('partnerDescriptionLink')}
                        onClick={this.showPartnerModal}
                        theme="normal"
                    >
                        {i18nBlock.partnerDescription()}
                    </LinkButton>
                </p>
            </div>
        );
    }

    private renderAddressAndRating(): ReactNode {
        const {hotelInfo, deviceType} = this.props;

        return (
            <HotelAddressAndRating
                className={cx('section', `section_size_m`)}
                hotelInfo={hotelInfo}
                onAddressClick={this.showPartnerModal}
                addressType="link"
                deviceType={deviceType}
                multiLine
            />
        );
    }

    private renderSearchParams(): ReactNode {
        const {hotelInfo, searchParams, deviceType} = this.props;

        const {isDesktop, isMobile} = deviceType;

        return (
            <BookSearchParams
                className={cx('section', {
                    section_size_l: isDesktop,
                    section_size_m: isMobile,
                })}
                hotelInfo={hotelInfo}
                searchParams={searchParams}
                showChangeParamsLink={false}
                {...prepareQaAttributes(this.props)}
            />
        );
    }

    private renderAmenities(): ReactNode {
        const {
            roomInfo: {amenities},
        } = this.props;

        return (
            <BookHotelAmenities
                className={cx('section', 'section_size_xs')}
                amenities={amenities}
                {...prepareQaAttributes({
                    parent: this.props,
                    current: 'amenities',
                })}
            />
        );
    }

    private renderMealInfo(): ReactNode {
        const {mealInfo, deviceType} = this.props;
        const {isDesktop, isMobile} = deviceType;
        const isSuccess =
            mealInfo.id !== EPansionType.RO &&
            mealInfo.id !== EPansionType.UNKNOWN;

        return (
            <BookMealInfo
                className={cx('section', {
                    section_size_l: isDesktop,
                    section_size_m: isMobile,
                })}
                mealInfo={mealInfo}
                isSuccess={isSuccess}
                {...prepareQaAttributes({
                    parent: this.props,
                    current: 'mealInfo',
                })}
            />
        );
    }

    private renderRoomImage(): ReactNode {
        const {
            roomInfo: {images},
        } = this.props;

        const showRoomImage = Boolean(images && images.length);

        return (
            showRoomImage && (
                <Image
                    className={cx('roomImageContainer')}
                    imageClassName={cx('roomImage')}
                    isCrop={true}
                    isWide={true}
                    src={getImagesBySize(images, THUMBNAIL_IMAGE_SIZE)[0].src}
                />
            )
        );
    }

    private renderRoomDescription(): ReactNode {
        const {roomInfo, deviceType} = this.props;

        return (
            <div className={cx('section', 'section_size_xs')}>
                <BookRoomDescription
                    deviceType={deviceType}
                    roomInfo={roomInfo}
                    showDescription={false}
                    {...prepareQaAttributes(this.props)}
                />
            </div>
        );
    }

    private renderSettlementInfo(): ReactNode {
        const {
            partnerId,
            hotelInfo: {settlementInfo},
            deviceType: {isMobile},
        } = this.props;

        return (
            <BookSettlementInfo
                isMobile={isMobile}
                partnerId={partnerId}
                className={cx('settlementInfo')}
                settlementInfo={settlementInfo}
                {...prepareQaAttributes({
                    parent: this.props,
                    current: 'settlementInfo',
                })}
            />
        );
    }

    private renderBedsGroups(): ReactNode {
        const {
            bedsGroups,
            setBedsGroupId,
            bookUserSelect: {bedsGroupId},
            deviceType: {isDesktop, isMobile},
        } = this.props;

        return (
            <BookSelectBedsGroups
                className={cx('section', {
                    section_size_l: isDesktop,
                    section_size_m: isMobile,
                })}
                setBedsGroupId={setBedsGroupId}
                bedsGroups={bedsGroups}
                bedsGroupId={bedsGroupId}
                {...prepareQaAttributes({
                    parent: this.props,
                    current: 'bedsGroups',
                })}
            />
        );
    }

    private renderRefundableInfo(): ReactNode {
        const {
            cancellationInfo,
            deviceType: {isDesktop, isMobile},
        } = this.props;

        if (!cancellationInfo) {
            return null;
        }

        return (
            <>
                <Text
                    size="m"
                    color="primary"
                    weight="bold"
                    tag="div"
                    className={cx('refundableTitle')}
                >
                    {i18nBlock.refundableRulesTitle()}
                </Text>

                <HotelsCancellationInfo
                    className={cx('section', {
                        section_size_l: isDesktop,
                        section_size_m: isMobile,
                    })}
                    labelTextSize="m"
                    cancellationInfo={cancellationInfo}
                    formatLabel={EFormatLabel.FULL_TEXT_WITH_DATE_AND_TIME}
                    isLabelPenaltyWithPrice
                    {...prepareQaAttributes({
                        parent: this.props,
                        current: 'cancellationInfo',
                    })}
                />
            </>
        );
    }

    render(): ReactNode {
        const {className, deviceType} = this.props;

        return (
            <div
                className={cx(
                    'content',
                    deviceMods('content', deviceType),
                    className,
                )}
            >
                {this.renderPartnerModal()}
                <TwoColumnLayout
                    deviceType={deviceType}
                    rightColumnOffset={5}
                    rightColumnWidth={50}
                >
                    <TwoColumnLayout.LeftColumn>
                        {this.renderName()}
                        {deviceType.isDesktop && this.renderAddressAndRating()}
                        {this.renderSearchParams()}
                        {this.renderRoomDescription()}
                        {this.renderAmenities()}
                        {this.renderMealInfo()}
                        {this.renderBedsGroups()}
                        {this.renderRefundableInfo()}
                        {this.renderSettlementInfo()}
                    </TwoColumnLayout.LeftColumn>
                    <TwoColumnLayout.RightColumn className={cx('rightColumn')}>
                        {deviceType.isDesktop && this.renderRoomImage()}
                        {deviceType.isDesktop &&
                            this.renderPartnerNameAndLink()}

                        {deviceType.isMobile && <Separator margin={3} />}
                        {deviceType.isMobile && this.renderPartnerNameAndLink()}
                    </TwoColumnLayout.RightColumn>
                </TwoColumnLayout>
            </div>
        );
    }
}

export default OfferHotelInfo;
