import isEmpty from 'lodash/isEmpty';
import {useRouteMatch} from 'react-router-dom';
import {MutableRefObject} from 'react';
import isEqual from 'lodash/isEqual';

import {
    IHotelsSearchFormErrors,
    IHotelsSearchFormPointField,
    IHotelsSearchFormValues,
    THotelsSearchFormDateField,
} from 'projects/hotels/components/SearchForm/types';
import {ESearchFormFieldName} from 'components/SearchForm/types';
import {HotelSlugType} from 'types/hotels/hotel/IHotel';
import {IHotelSlugRouteMatch} from 'types/hotels/hotel/IHotelSlugRouteMatch';
import {EIndexGoal} from 'utilities/metrika/types/goals';
import {IHotelsPreviousSearch} from 'types/hotels/previousSearches/IPreviousSearch';
import {EHotelsGoal} from 'utilities/metrika/types/goals/hotels';

import {CustomDispatch} from 'reducers/trains/customDispatch';
import addPreviousSearchThunkAction from 'reducers/hotels/previousSearches/thunk/addPreviousSearchThunkAction';

import useImmutableCallback from 'utilities/hooks/useImmutableCallback';
import {reachGoal} from 'utilities/metrika';
import {getHotelSlugByRouteMatch} from 'projects/hotels/utilities/getHotelIdentifier/getHotelIdentifier';
import {hotelsURLs} from 'projects/hotels/utilities/urls';
import history from 'utilities/browserHistory/browserHistory';
import {isHotelsCrossSearch} from 'utilities/searchForm/isCrossSearch';

export interface ISubmitArgs {
    handleTrySubmit(): void;
    formErrors: IHotelsSearchFormErrors;
    setHasSubmitFailed(val: boolean): void;
}

export default function useHandleSubmit({
    toField,
    startDateField,
    endDateField,
    adultsField,
    childrenAgesField,
    activeHotelSlug,
    queryParams,
    hasProductionOffers,
    hasDirectionSuggest,
    hasDelayBeforeUpdateLocation,
    dispatch,
    chosenPreviousSearch,
    chosenPersonalSuggest,
    onSubmit,
}: {
    toField: IHotelsSearchFormPointField;
    startDateField: THotelsSearchFormDateField;
    endDateField: THotelsSearchFormDateField;
    adultsField: number;
    childrenAgesField: number[];
    activeHotelSlug: HotelSlugType | undefined;
    queryParams?: Record<string, string>;
    hasProductionOffers: boolean;
    hasDirectionSuggest: boolean;
    hasDelayBeforeUpdateLocation: boolean;
    dispatch: CustomDispatch;
    chosenPreviousSearch: MutableRefObject<IHotelsPreviousSearch | null>;
    chosenPersonalSuggest: MutableRefObject<IHotelsPreviousSearch | null>;
    onSubmit?(params: {searchLink: string}): void;
}): (args: ISubmitArgs) => void {
    const match = useRouteMatch<IHotelSlugRouteMatch>();

    const openSearchLink = useImmutableCallback((searchLink: string) => {
        history?.push(searchLink, {scrollTop: null});
    });

    return useImmutableCallback(
        ({handleTrySubmit, formErrors, setHasSubmitFailed}) => {
            reachGoal(EIndexGoal.HOTELS_SEARCH_BUTTON);

            handleTrySubmit();

            if (isEmpty(formErrors)) {
                const formValues: IHotelsSearchFormValues = {
                    [ESearchFormFieldName.TO]: toField,
                    [ESearchFormFieldName.START_DATE]: startDateField,
                    [ESearchFormFieldName.END_DATE]: endDateField,
                    adults: adultsField,
                    childrenAges: childrenAgesField,
                };

                const hotelSlugByRouteMatch = getHotelSlugByRouteMatch(match);

                const isChosenPersonalSuggest = isEqual(
                    chosenPersonalSuggest.current,
                    {
                        to: toField.selectedValue,
                        startDate: startDateField,
                        endDate: endDateField,
                        adults: adultsField,
                        childrenAges: childrenAgesField,
                    },
                );

                if (isChosenPersonalSuggest) {
                    chosenPersonalSuggest.current = null;
                }

                const searchLink = hotelsURLs.getSearchUrl(
                    {
                        activeHotelSlug,
                        hotelSlugByRouteMatch,
                        searchFormFieldsValues: formValues,
                        queryParams: isChosenPersonalSuggest
                            ? {
                                  ...queryParams,
                                  ytp_referer: 'hotels_suggest_crossale',
                              }
                            : queryParams,
                    },
                    {
                        hasProductionOffers,
                        hasDirectionSuggest,
                    },
                );

                onSubmit?.({searchLink});

                /* hasDelayBeforeUpdateLocation - for improve form animation */
                if (hasDelayBeforeUpdateLocation) {
                    requestAnimationFrame(() => {
                        openSearchLink(searchLink);
                    });
                } else {
                    openSearchLink(searchLink);
                }

                if (toField.selectedValue) {
                    const newPreviousSearch = {
                        to: toField.selectedValue,
                        startDate: startDateField,
                        endDate: endDateField,
                        adults: adultsField,
                        childrenAges: childrenAgesField,
                    };

                    dispatch(addPreviousSearchThunkAction(newPreviousSearch));

                    if (
                        isEqual(chosenPreviousSearch.current, newPreviousSearch)
                    ) {
                        reachGoal(EHotelsGoal.PREVIOUS_SEARCHES_SUGGEST_SEARCH);

                        chosenPreviousSearch.current = null;
                    }
                }

                if (isHotelsCrossSearch(toField)) {
                    reachGoal(EHotelsGoal.CROSS_SEARCH_SUBMIT);
                }
            } else {
                setHasSubmitFailed(true);
            }
        },
    );
}
