import {Component, ReactNode} from 'react';
import {withRouter, RouteComponentProps} from 'react-router-dom';
import {connect} from 'react-redux';

/* Types */
import {IHotelWithOffers} from 'types/hotels/hotel/IHotelWithOffers';
import {ISimilarHotelsInfo} from 'reducers/hotels/hotelPage/hotelInfo/types';
import {IWithClassName} from 'src/types/withClassName';
import {IRequiredOfferParams} from 'types/hotels/offer/IHotelOffer';
import {HotelSlugType} from 'types/hotels/hotel/IHotel';
import EHotelImageCustomSize from 'projects/hotels/pages/HotelPage/components/Gallery/types/EHotelImageCustomSize';
import {IWithDeviceType} from 'types/withDeviceType';

/* Utilities */
import {StoreInterface} from 'reducers/storeTypes';

import experimentsSelector from 'selectors/common/experimentsSelector';

import getQueryByLocation from 'utilities/getQueryByLocation/getQueryByLocation';
import {prepareQaAttributes} from 'utilities/qaAttributes/qaAttributes';
import {hotelsURLs} from 'projects/hotels/utilities/urls';
import {deviceMods} from 'utilities/stylesUtils';

import * as hotelPageKeyset from 'i18n/hotels-HotelPage';

/* Components */
import Link from 'components/Link/Link';
import HotelShortView from 'projects/hotels/components/HotelShortView/HotelShortView';
import SimilarHotelSkeleton from './components/SimilarHotelSkeleton/SimilarHotelSkeleton';

/* Styles */
import cx from './SimilarHotels.scss';

interface ISimilarHotelsProps
    extends IWithClassName,
        RouteComponentProps,
        ISimilarHotelsSelector,
        IWithDeviceType {
    similarHotelsInfo: ISimilarHotelsInfo;
    isMobile?: boolean;
    onClickCard?: () => void;
    searchParams?: Partial<IRequiredOfferParams>;
}

interface ISimilarHotelsSelector {
    hotelsPercentDiscount: boolean;
}

class SimilarHotels extends Component<ISimilarHotelsProps> {
    /* Handlers */

    private handleClickHotelCard = (): void => {
        const {onClickCard} = this.props;

        if (onClickCard) {
            onClickCard();
        }
    };

    /* Render */

    private renderCardOverlay(hotelSlug: HotelSlugType): ReactNode {
        const {searchParams, location} = this.props;
        const queryByLocation = getQueryByLocation(location);

        const hotelLandingUrl = hotelsURLs.getHotelUrl(
            {
                hotelSlug,
                ...searchParams,
            },
            queryByLocation,
        );

        return (
            <Link
                url={hotelLandingUrl}
                className={cx('hotelLink')}
                onClick={this.handleClickHotelCard}
                target="_blank"
                rel="noopener noreferrer"
            />
        );
    }

    private renderSimilarHotel = (
        similarHotel: IHotelWithOffers,
        index: number,
    ): ReactNode => {
        const {
            similarHotelsInfo,
            isMobile,
            searchParams,
            location,
            hotelsPercentDiscount,
        } = this.props;
        const {
            hotel: {permalink, hotelSlug},
        } = similarHotel;
        const isFinished = similarHotelsInfo.offerSearchProgress.finished;
        const queryByLocation = getQueryByLocation(location);

        const hotelViewNode = isFinished ? (
            <HotelShortView
                key={isMobile ? undefined : permalink}
                className={cx('hotel')}
                onClickHotelName={this.handleClickHotelCard}
                titleClassName={cx('hotelName')}
                queryByLocation={queryByLocation}
                canRenderOfferPrice
                hotel={similarHotel.hotel}
                offers={similarHotel.offers}
                searchParams={searchParams}
                imageSizeName={EHotelImageCustomSize.S1}
                hotelsPercentDiscount={hotelsPercentDiscount}
                {...prepareQaAttributes({current: 'similarHotel', key: index})}
            />
        ) : (
            <SimilarHotelSkeleton className={cx('hotel')} />
        );

        return (
            <div className={cx('hotelBlock')} key={permalink}>
                {hotelViewNode}
                {this.renderCardOverlay(hotelSlug)}
            </div>
        );
    };

    render(): ReactNode {
        const {similarHotelsInfo, className, deviceType} = this.props;
        const {isMobile} = deviceType;

        return (
            <div className={cx({root_mobile: isMobile}, className)}>
                <div className={cx('title', deviceMods('title', deviceType))}>
                    {hotelPageKeyset.similarHotels()}
                </div>
                <div>
                    {similarHotelsInfo.hotels.map(this.renderSimilarHotel)}
                </div>
            </div>
        );
    }
}

export default withRouter(
    connect((state: StoreInterface) => ({
        hotelsPercentDiscount: Boolean(
            experimentsSelector(state).hotelsPercentDiscount,
        ),
    }))(SimilarHotels),
);
