import {FunctionComponent, useContext, useRef, useMemo} from 'react';

import {IHotelOrOfferBadge} from 'types/hotels/offer/IHotelOffer';
import EPopupDirection from 'components/Popup/types/EPopupDirection';

import {deviceModDesktop} from 'utilities/stylesUtils';
import {useBoolean} from 'utilities/hooks/useBoolean';
import {useDeviceType} from 'utilities/hooks/useDeviceType';

import Link from 'components/Link/Link';
import Text from 'components/Text/Text';
import YandexEdaIcon from 'icons/12/YandexEda';
import Heading from 'components/Heading/Heading';
import LinkButton from 'components/LinkButton/LinkButton';
import TextWithIcon from 'components/TextWithIcon/TextWithIcon';
import {EMessageBoxPopupTheme} from 'components/MessageBoxPopup/MessageBoxPopup';
import MessageBoxPopupOrBottomSheet from 'components/MessageBoxPopupOrBottomSheet/MessageBoxPopupOrBottomSheet';

import ScopeContext from 'contexts/ScopeContext';

import cx from './YandexEdaPromo.scss';

const POPUP_DIRECTION = EPopupDirection.TOP;

interface IYandexEdaPromoProps {
    badge: IHotelOrOfferBadge;
}

const YandexEdaPromo: FunctionComponent<IYandexEdaPromoProps> = ({
    badge: {text, additionalPromoInfo},
}) => {
    const deviceType = useDeviceType();
    const scope = useContext(ScopeContext);
    const buttonRef = useRef<HTMLElement>(null);
    const {value: isVisible, setTrue: show, setFalse: hide} = useBoolean(false);

    const messageBoxProps = useMemo(() => {
        return {
            theme: EMessageBoxPopupTheme.WHITE,
            direction: POPUP_DIRECTION,
            anchorRef: buttonRef,
            scopeRef: scope ? {current: scope} : undefined,
        };
    }, [scope]);

    return (
        <>
            <LinkButton
                className={cx('root')}
                controlRef={buttonRef}
                theme="black"
                onClick={show}
            >
                <TextWithIcon
                    text={text}
                    textClassName={cx('labelText')}
                    iconLeft={YandexEdaIcon}
                    iconLeftClassName={cx('labelIcon')}
                    size="s-inset"
                    iconSize={12}
                    spaceBetween={1}
                />
            </LinkButton>
            {additionalPromoInfo && (
                <MessageBoxPopupOrBottomSheet
                    isVisible={isVisible}
                    onClose={hide}
                    messageBoxProps={messageBoxProps}
                >
                    <div
                        className={cx(deviceModDesktop('content', deviceType))}
                    >
                        {additionalPromoInfo.title && (
                            <Heading level="3" className={cx('title')}>
                                {additionalPromoInfo.title}
                            </Heading>
                        )}
                        <Text size="m" tag="div">
                            {additionalPromoInfo.text}
                        </Text>
                        {additionalPromoInfo.link && (
                            <Text className={cx('link')} size="m" tag="div">
                                <Link
                                    url={additionalPromoInfo.link.url}
                                    target="_blank"
                                >
                                    {additionalPromoInfo.link.text}
                                </Link>
                            </Text>
                        )}
                    </div>
                </MessageBoxPopupOrBottomSheet>
            )}
        </>
    );
};

export default YandexEdaPromo;
