import React, {Component} from 'react';
import _noop from 'lodash/noop';
import _isUndefined from 'lodash/isUndefined';

import {
    CANCELLED,
    CANCELLED_WITH_REFUND,
    FAILED,
    INITIAL,
    NEED_AUTHORIZATION,
    UNKNOWN_ORDER_ERROR,
} from 'projects/hotels/constants/hotelsBookingStatuses';
import {OrderCancellationDetailsReasonType} from 'server/api/HotelsBookAPI/constants/orderCancellationDetails';
import {EOfferStatus} from 'projects/hotels/constants/EOfferStatus';

import {EHotelsGoal} from 'utilities/metrika/types/goals/hotels';

import {getReason} from 'selectors/hotels/book/orderCancellationDetails/getReason';

import getQueryByLocation from 'utilities/getQueryByLocation/getQueryByLocation';
import {deviceMods} from 'utilities/stylesUtils';
import {reachGoal} from 'utilities/metrika';
import {prepareQaAttributes} from 'utilities/qaAttributes/qaAttributes';
import {insertJSXIntoKey} from 'utilities/tanker/insertJSXIntoKey';

import * as i18nBlock from 'i18n/hotels-BookErrorPage';

import Link from 'components/Link/Link';
import BookLoader from 'components/BookLoader/BookLoader';
import PortalServerError from 'components/PortalServerError/PortalServerError';
import BookHotelInfo from 'projects/hotels/components/BookHotelInfo/BookHotelInfo';
import CheckOrderAuthorization from 'components/CheckOrderAuthorization/CheckOrderAuthorizationContainer';
import HotelsBookLayout from 'projects/hotels/components/HotelsBookLayout/HotelsBookLayout';
import BookErrorSearchHotelsButton from 'projects/hotels/pages/BookErrorPage/components/BookErrorSearchHotelsButton';
import BookErrorSearchHotelButton from 'projects/hotels/pages/BookErrorPage/components/BookErrorSearchHotelButton';

import BookRedirectController from 'projects/hotels/containers/BookRedirectController/BookRedirectController';
import bookInfoProvider, {
    IBookInfoComponentProps,
} from 'projects/hotels/containers/BookInfoProvider/BookInfoProvider';

import cx from './BookErrorPage.scss';

const QA_BOOK_ERROR_PAGE = 'hotelsBookErrorPage';

interface IBookErrorPageProps extends IBookInfoComponentProps {}

class BookErrorPage extends Component<IBookErrorPageProps> {
    static defaultProps = {
        offerInfoByToken: {
            isLoading: false,
        },
        deviceType: {},
        orderInfo: {},
        startOrderPolling: _noop,
        fetchOfferInfoByToken: _noop,
    };

    componentDidMount(): void {
        const {
            fetchOfferInfoByToken,
            location,
            orderInfo: {status: orderStatus},
            offerInfoByToken: {status: offerStatus},
        } = this.props;

        const {token, label, orderId} = getQueryByLocation(location, false);

        /* Fetch offer info for empty orderId by token and label from query params */
        if (
            orderStatus === INITIAL &&
            _isUndefined(orderId) &&
            !offerStatus &&
            token
        ) {
            fetchOfferInfoByToken({token, label: label ?? undefined});
        }
    }

    /* Helpers */

    checkRenderLoader = (): boolean => {
        const {
            location,
            orderInfo: {status},
            offerInfoByToken: {isLoading},
        } = this.props;
        const {orderId} = getQueryByLocation(location);

        return Boolean(isLoading || (orderId && status === INITIAL));
    };

    startOrderPolling = (): void => {
        const {location, startOrderPolling} = this.props;
        const {orderId} = getQueryByLocation(location, false);

        reachGoal(EHotelsGoal.ORDER_AUTH_SUCCESS);

        if (!orderId) {
            return;
        }

        startOrderPolling({orderId});
    };

    handleFailedAuthorization = (): void => {
        reachGoal(EHotelsGoal.ORDER_AUTH_FAILED);
    };

    /* Loaders */

    renderPageLoader(): React.ReactNode {
        const canRenderPageLoader = this.checkRenderLoader();

        return (
            <BookLoader
                title={i18nBlock.offerLoadingDotTitle()}
                description={i18nBlock.offerLoadingDotDescription()}
                isLoading={canRenderPageLoader}
                {...prepareQaAttributes({
                    current: 'loader',
                })}
            />
        );
    }

    /* Hotel Info main card */

    renderHotelInfo(): React.ReactNode {
        const {
            offerInfoByToken: {offerInfo},
            orderInfo: {status: orderStatus},
            deviceType,
        } = this.props;

        if (!offerInfo) {
            return null;
        }

        const {
            roomInfo,
            hotelInfo,
            bedsGroups,
            searchParams,
            cancellationInfo,
            partnerHotelInfo,
            mealInfo,
        } = offerInfo;

        const firstBedsGroup = bedsGroups.slice(0, 1);
        const canRenderSearchButtons = orderStatus !== FAILED;
        const footerNode = canRenderSearchButtons && this.renderSearchButtons();

        return (
            <BookHotelInfo
                className={cx('bookHotelInfo', 'bookPageBlock')}
                deviceType={deviceType}
                roomInfo={roomInfo}
                hotelInfo={hotelInfo}
                mealInfo={mealInfo}
                searchParams={searchParams}
                bedsGroups={firstBedsGroup}
                partnerHotelInfo={partnerHotelInfo}
                cancellationInfo={cancellationInfo}
                footerNode={footerNode}
                isExpiredOffer
            />
        );
    }

    renderHotelInfoWithErrorTitle(title: string): React.ReactNode {
        return (
            <div className={cx('errorContent')}>
                <h3
                    className={cx('errorTitle')}
                    {...prepareQaAttributes('errorTitle')}
                >
                    {title}
                </h3>
                {this.renderHotelInfo()}
            </div>
        );
    }

    /* Render blocks by error type */

    renderOfferErrorByStatus(status: string): React.ReactNode {
        switch (status) {
            case EOfferStatus.OFFER_IS_NOT_AVAILABLE: {
                return this.renderExpiredOfferInfo();
            }

            case EOfferStatus.UNKNOWN_OFFER_ERROR: {
                return this.renderUnknownServerError();
            }
        }
    }

    renderOrderErrorByStatus(status: string): React.ReactNode {
        switch (status) {
            case CANCELLED: {
                return this.renderCancelledOrder();
            }

            case FAILED: {
                return this.renderFailedOrder();
            }

            case CANCELLED_WITH_REFUND: {
                return this.renderCancelledWithRefundOrder();
            }

            case UNKNOWN_ORDER_ERROR: {
                return this.renderUnknownServerError();
            }

            case NEED_AUTHORIZATION: {
                return this.renderCheckOrderAuthorization();
            }
        }
    }

    renderExpiredOfferInfo(): React.ReactNode {
        return this.renderHotelInfoWithErrorTitle(
            i18nBlock.expiredOfferTitle(),
        );
    }

    renderUnknownServerError(): React.ReactNode {
        return <PortalServerError />;
    }

    renderCancelledOrder(): React.ReactNode {
        const reason = getReason(this.props.orderInfo.orderCancellationDetails);

        let title;

        switch (reason) {
            case OrderCancellationDetailsReasonType.SOLD_OUT: {
                title = i18nBlock.expiredOfferTitle();

                break;
            }
            case OrderCancellationDetailsReasonType.PRICE_CHANGED: {
                title = i18nBlock.cancelledOrderTitleDotPriceChanged();

                break;
            }

            default: {
                title = i18nBlock.cancelledOrderTitle();

                break;
            }
        }

        return this.renderHotelInfoWithErrorTitle(title);
    }

    renderCheckOrderAuthorization(): React.ReactNode {
        const {location} = this.props;
        const {orderId} = getQueryByLocation(location, false);

        if (!orderId) {
            return null;
        }

        return (
            <CheckOrderAuthorization
                className={cx('checkOrderAuthorization')}
                orderId={orderId}
                onSuccessOrderAuthorization={this.startOrderPolling}
                onFailedOrderAuthorization={this.handleFailedAuthorization}
            />
        );
    }

    renderSupportInfo(): React.ReactNode {
        const {
            orderInfo: {confirmationInfo},
        } = this.props;
        const yandexOrderId = confirmationInfo?.yandexOrderId;

        const supportLinkNode = (
            <Link
                url={i18nBlock.supportLinkHref()}
                text={i18nBlock.supportLinkTitle()}
                target="_blank"
                rel="noopener noreferrer"
                tabIndex={-1}
                {...prepareQaAttributes({
                    current: 'supportLink',
                })}
            />
        );

        const orderIdNode = (
            <span
                className={cx('errorOrderId')}
                {...prepareQaAttributes({
                    current: 'errorOrderId',
                })}
            >
                {yandexOrderId}
            </span>
        );

        return (
            <>
                <div className={cx('errorSectionItem')}>
                    {insertJSXIntoKey(i18nBlock.supportFullText)({
                        supportLinkNode,
                    })}
                </div>
                <div className={cx('errorSectionItem')}>
                    {insertJSXIntoKey(i18nBlock.supportOrderId)({
                        orderId: orderIdNode,
                    })}
                </div>
            </>
        );
    }

    renderFailedOrder(): React.ReactNode {
        return (
            <div className={cx('errorContent')}>
                <section className={cx('errorSection')}>
                    <h4
                        className={cx('errorSectionTitle')}
                        {...prepareQaAttributes({
                            current: 'errorSectionTitle',
                        })}
                    >
                        {i18nBlock.failedOrderDotTitle()}
                    </h4>
                    <div className={cx('errorSectionDescription')}>
                        {i18nBlock.failedOrderDotDescription()}
                    </div>
                    <div className={cx('errorSectionItem')}>
                        {i18nBlock.failedOrderDotSupportTitle()}
                    </div>
                    {this.renderSupportInfo()}
                </section>
                {this.renderHotelInfo()}
            </div>
        );
    }

    renderCancelledWithRefundOrder(): React.ReactNode {
        return (
            <div className={cx('errorContent')}>
                <section className={cx('errorSection')}>
                    <h4
                        className={cx('errorSectionTitle')}
                        {...prepareQaAttributes({
                            current: 'errorSectionTitle',
                        })}
                    >
                        {i18nBlock.cancelledWithRefundDotTitle()}
                    </h4>
                    <div className={cx('errorSectionDescription')}>
                        {i18nBlock.cancelledWithRefundDotDescription()}
                    </div>
                    <div className={cx('errorSectionItem')}>
                        {i18nBlock.cancelledWithRefundDotRefundMoneyTitle()}
                    </div>
                    {this.renderSupportInfo()}
                </section>
                {this.renderHotelInfo()}
            </div>
        );
    }

    /* Render new search buttons */

    renderSearchButtons = (): React.ReactNode => {
        const {
            offerInfoByToken: {offerInfo},
        } = this.props;

        return (
            <div className={cx('searchButtons')}>
                <BookErrorSearchHotelsButton offerInfo={offerInfo} />
                <BookErrorSearchHotelButton offerInfo={offerInfo} />
            </div>
        );
    };

    /* Main Content Block */

    render(): React.ReactNode {
        const {
            deviceType,
            offerInfoByToken: {status: offerStatus},
            orderInfo: {status: orderStatus},
        } = this.props;

        const errorNode =
            this.renderOfferErrorByStatus(offerStatus) ||
            this.renderOrderErrorByStatus(orderStatus);
        const canRenderPageLoader = this.checkRenderLoader();

        return (
            <HotelsBookLayout
                deviceType={deviceType}
                hasLoader={canRenderPageLoader}
            >
                <BookRedirectController />
                <div
                    className={cx(
                        'bookErrorPage',
                        deviceMods('bookErrorPage', deviceType),
                    )}
                    {...prepareQaAttributes(QA_BOOK_ERROR_PAGE)}
                >
                    {this.renderPageLoader()}
                    <div className={cx('content')}>{errorNode}</div>
                </div>
            </HotelsBookLayout>
        );
    }
}

export default bookInfoProvider({
    canFetchOfferInfo: false,
    canStartOrderPolling: true,
})(BookErrorPage);
