import {FC, MouseEventHandler, useCallback, useMemo} from 'react';

import {EAppActions} from 'constants/platforms/TPlatforms';

import {IBookOffer} from 'server/api/HotelsBookAPI/types/IBookOffer';

import {prepareQaAttributes} from 'utilities/qaAttributes/qaAttributes';
import {getHotelUrl} from 'projects/hotels/utilities/urls/getHotelUrl';

import * as i18nBlock from 'i18n/hotels-BookErrorPage';

import ButtonLink from 'components/ButtonLink/ButtonLink';
import {ContainerElement} from '@yandex-lego/components/Link/Link';

import cx from 'projects/hotels/pages/BookErrorPage/BookErrorPage.scss';

import {useCoordinator} from 'contexts/PlatformContext';

export interface IBookErrorSearchHotelButtonProps {
    offerInfo?: IBookOffer;
}

const BookErrorSearchHotelButton: FC<IBookErrorSearchHotelButtonProps> = ({
    offerInfo,
}) => {
    const coordinator = useCoordinator();

    const url = useMemo<string | undefined>(() => {
        const hotelSlug = offerInfo?.hotelInfo.hotelSlug;
        const searchParams = offerInfo?.searchParams;

        if (!hotelSlug || !searchParams) {
            return;
        }

        return getHotelUrl({
            ...searchParams,
            hotelSlug,
            checkinDate: searchParams.startDate,
            checkoutDate: searchParams.endDate,
        });
    }, [offerInfo?.hotelInfo.hotelSlug, offerInfo?.searchParams]);

    const handleOnClick = useCallback<MouseEventHandler<ContainerElement>>(
        event => {
            const hotelSlug = offerInfo?.hotelInfo.hotelSlug;
            const searchParams = offerInfo?.searchParams;

            if (!hotelSlug || !searchParams) {
                return;
            }

            const processed = coordinator.doAction(
                EAppActions.NAVIGATE_TO_VARIANT_HOTELS_PAGE,
                {
                    hotelSlug,
                    searchParams: {
                        adults: searchParams.adults,
                        childrenAge: searchParams.childrenAges,
                        checkinDate: searchParams.startDate,
                        checkoutDate: searchParams.endDate,
                    },
                    url,
                },
            );

            if (processed) {
                event.preventDefault();
            }
        },
        [
            coordinator,
            offerInfo?.hotelInfo.hotelSlug,
            offerInfo?.searchParams,
            url,
        ],
    );

    return (
        <ButtonLink
            className={cx('searchButton')}
            theme="primary"
            url={url}
            {...prepareQaAttributes({
                current: 'searchHotelButton',
            })}
            onClick={handleOnClick}
        >
            <span className={cx('searchButtonText')}>
                {i18nBlock.searchHotelButton()}
            </span>
        </ButtonLink>
    );
};

export default BookErrorSearchHotelButton;
