import {FC, MouseEventHandler, useCallback, useMemo} from 'react';

import {EAppActions} from 'constants/platforms/TPlatforms';

import {IBookOffer} from 'server/api/HotelsBookAPI/types/IBookOffer';

import {prepareQaAttributes} from 'utilities/qaAttributes/qaAttributes';
import {getHotelsSearchUrlByParams} from 'projects/hotels/utilities/urls/getHotelsSearchUrl';

import * as i18nBlock from 'i18n/hotels-BookErrorPage';

import ButtonLink from 'components/ButtonLink/ButtonLink';
import {ContainerElement} from '@yandex-lego/components/Link/Link';

import cx from 'projects/hotels/pages/BookErrorPage/BookErrorPage.scss';

import {useCoordinator} from 'contexts/PlatformContext';

export interface IBookErrorSearchHotelsButtonProps {
    offerInfo?: IBookOffer;
}

const BookErrorSearchHotelsButton: FC<IBookErrorSearchHotelsButtonProps> = ({
    offerInfo,
}) => {
    const coordinator = useCoordinator();

    const url = useMemo(() => {
        const geoRegions = offerInfo?.hotelInfo.breadcrumbs?.geoRegions;
        const searchParams = offerInfo?.searchParams;

        if (!geoRegions || !searchParams) {
            return;
        }

        return getHotelsSearchUrlByParams(
            geoRegions[geoRegions.length - 1].geoId,
            {
                ...searchParams,
                checkinDate: searchParams.startDate,
                checkoutDate: searchParams.endDate,
            },
        );
    }, [offerInfo?.hotelInfo.breadcrumbs?.geoRegions, offerInfo?.searchParams]);

    const handleOnClick = useCallback<MouseEventHandler<ContainerElement>>(
        event => {
            const geoRegions = offerInfo?.hotelInfo.breadcrumbs?.geoRegions;
            const searchParams = offerInfo?.searchParams;

            if (!geoRegions || !searchParams) {
                return;
            }

            const processed = coordinator.doAction(
                EAppActions.NAVIGATE_TO_SEARCH_RESULT_HOTELS_PAGE,
                {
                    region: geoRegions[geoRegions.length - 1],
                    searchParams: {
                        adults: searchParams.adults,
                        childrenAge: searchParams.childrenAges,
                        checkinDate: searchParams.startDate,
                        checkoutDate: searchParams.endDate,
                    },
                },
            );

            if (processed) {
                event.preventDefault();
            }
        },
        [
            coordinator,
            offerInfo?.hotelInfo.breadcrumbs?.geoRegions,
            offerInfo?.searchParams,
        ],
    );

    return (
        <ButtonLink
            url={url}
            onClick={handleOnClick}
            {...prepareQaAttributes({
                current: 'searchHotelsButton',
            })}
        >
            <span className={cx('searchButtonText')}>
                {i18nBlock.searchHotelsButton()}
            </span>
        </ButtonLink>
    );
};

export default BookErrorSearchHotelsButton;
