import React, {ReactNode} from 'react';
import _noop from 'lodash/noop';
import omit from 'lodash/omit';
import setFieldData from 'final-form-set-field-data';

import {
    INITIAL,
    PAYMENT_FAILED,
} from 'projects/hotels/constants/hotelsBookingStatuses';
import {URLs} from 'constants/urls';
import {EProjectName} from 'constants/common';
import {ERedirectStatusCodes} from 'constants/redirectStatusCodes';
import {EFormKey} from 'constants/form/EFormKey';
import {SOLOMON_HOTELS_PAY_BUTTON_CLICK} from 'constants/solomon';
import {EOfferStatus} from 'projects/hotels/constants/EOfferStatus';

import {EHotelsGoal} from 'utilities/metrika/types/goals/hotels';
import {IWithDeviceType} from 'types/withDeviceType';
import {
    EYandexPlusApplicationMode,
    IAppliedPromoCampaigns,
} from 'types/hotels/offer/IHotelOffer';
import {
    EBookGuestFieldName,
    EFormGroup,
    IBookContacts,
    IBookFormValues,
    IBookPriceInfo,
    IBookWhiteLabel,
    IStorageBookGuest,
} from 'types/hotels/book/IBookFormValues';
import {IWithWhiteLabelConfig} from 'types/common/whiteLabel/IWithWhiteLabelConfig';

import {IOfferInfoByTokenReducer} from 'reducers/hotels/bookAndPayPage/offerInfoByToken/reducer';

import {getSessionKey} from 'selectors/hotels/book/getSessionKey';

import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {internalUrl} from 'utilities/url';
import {params, reachGoal} from 'utilities/metrika';
import {getBookingContacts} from 'utilities/localStorage/booking/getBookingContacts';
import {getBookingPassengers} from 'utilities/localStorage/booking/getBookingPassengers';
import getQueryByLocation from 'utilities/getQueryByLocation/getQueryByLocation';
import {getHotelOfferToken} from 'projects/hotels/utilities/getHotelPageParams/getOfferAttributionParamsByLocation';
import getFinishPaymentPagePath from 'projects/hotels/utilities/getFinishPaymentPagePath/getFinishPaymentPagePath';
import getGuestsByOrderInfo from './utilities/getGuestsByOrderInfo';
import {sendPaymentCounter} from 'utilities/solomon';
import getTotalTopupPlusPoints from 'projects/hotels/pages/BookPage/utilities/getTotalTopupPlusPoints';
import getTotalWithdrawPlusPoints from 'projects/hotels/pages/BookPage/utilities/getTotalWithdrawPlusPoints';
import optionalAdultsValidationInfo from './utilities/validationInfo';

import * as i18nBlock from 'i18n/hotels-BookPayPage';

import Form from 'components/Form/Form';
import focusFirstInvalidField from 'components/Form/mutators/focusFirstInvalidField';
import createSubmitErrorDecorator from 'components/Form/decorators/createSubmitErrorDecorator';
import BookHeader, {
    BOOK_STEPS,
} from 'projects/hotels/components/BookHeader/BookHeader';
import BookLoader from 'components/BookLoader/BookLoader';
import HotelsBookLayout from 'projects/hotels/components/HotelsBookLayout/HotelsBookLayout';
import RedirectWithStatus from 'components/RedirectWithStatus/RedirectWithStatus';
import HangingYandexMessenger from 'components/YandexMessenger/components/HangingYandexMessenger/HangingYandexMessenger';

import BookRedirectController from 'projects/hotels/containers/BookRedirectController/BookRedirectController';

import FormController from './blocks/FormController/FormController';
import BookPageContent from './blocks/BookPageContent/BookPageContentContainer';
import {TBookPageContainerProps} from './BookPageContainer';

import cx from './BookPage.scss';

const HOTELS_BOOK_PAGE_QA = 'hotelsBookPage';

const validateOnSubmit = createSubmitErrorDecorator<IBookFormValues>(
    EFormKey.HOTEL_BOOK,
);

interface IBookPageProps
    extends TBookPageContainerProps,
        IWithDeviceType,
        IWithWhiteLabelConfig {}

interface IBookPageState {
    canRenderGoToForm: boolean;
    wasSessionKeyRenewed: boolean;
}

class BookPage extends React.PureComponent<IBookPageProps, IBookPageState> {
    private static getInitialWhiteLabel(): IBookWhiteLabel {
        return {
            customerNumber: '',
        };
    }

    private static isOfferBecomeSuccess(
        offerInfoByToken: IOfferInfoByTokenReducer,
        prevOfferInfoByToken: IOfferInfoByTokenReducer,
    ): boolean {
        return offerInfoByToken.isSuccess && !prevOfferInfoByToken.isSuccess;
    }

    static defaultProps = {
        offerInfoByToken: {
            isLoading: false,
        },
        deviceType: {},
        orderInfo: {},
        userInfo: {},
        bookUserSelect: {},
        fetchOfferInfoByToken: _noop,
        resetBookUserSelect: _noop,
        createOrder: _noop,
    };

    state = {
        canRenderGoToForm: true,
        // Флаг, что с бэка пришел новый sessionKey (нужно для повторной попытки брони)
        wasSessionKeyRenewed: false,
    };

    componentDidMount(): void {
        this.fetchOfferInfoByToken();
        this.props.fetchPassengers();

        reachGoal(EHotelsGoal.BOOK_PAGE_LOAD, {
            hotels: {
                didVisitBookingPage: 1,
            },
        });
    }

    componentDidUpdate(prevProps: IBookPageProps): void {
        const {
            offerInfoByToken,
            orderInfo: {status},
        } = this.props;
        const {offerInfoByToken: prevOfferInfoByToken} = prevProps;

        const isOfferSuccess = BookPage.isOfferBecomeSuccess(
            offerInfoByToken,
            prevOfferInfoByToken,
        );

        if (
            isOfferSuccess &&
            offerInfoByToken.status === EOfferStatus.OFFER_FETCHED
        ) {
            reachGoal(EHotelsGoal.BOOK_PAGE_SHOWN);
        }

        if (isOfferSuccess && status === PAYMENT_FAILED) {
            reachGoal(EHotelsGoal.BOOK_PAGE_ERROR, {
                hotels: {orderError: 'payment_error'},
            });
        }

        this.checkWasSessionKeyRenewed(prevProps);
    }

    private checkWasSessionKeyRenewed = (prevProps: IBookPageProps): void => {
        const prevSessionKey = getSessionKey(prevProps);
        const currentSessionKey = getSessionKey(this.props);

        if (currentSessionKey && currentSessionKey !== prevSessionKey) {
            this.setState({wasSessionKeyRenewed: true});
        }
    };

    private fetchOfferInfoByToken(): void {
        const {
            offerInfoByToken: {offerInfo},
            fetchOfferInfoByToken,
        } = this.props;

        /* Fetch offer info for update sessionKey */
        if (offerInfo?.offerOrderInfo) {
            const {token, label} = offerInfo.offerOrderInfo;

            fetchOfferInfoByToken({token, label});
        }
    }

    private getOfferToken = (): string => {
        const {location} = this.props;
        const query = getQueryByLocation(location);
        const {token} = getHotelOfferToken(query);

        return token || '';
    };

    private getInitialContacts(): IBookContacts {
        const {
            orderInfo: {guestsInfo},
            userInfo,
        } = this.props;

        const {email, phone} = getBookingContacts(
            EFormKey.HOTEL_BOOK,
            userInfo,
        );
        const {customerPhone: orderPhone, customerEmail: orderEmail} =
            guestsInfo || {};

        return {
            validationKey: 'contacts',
            email: orderEmail || email,
            phone: orderPhone || phone,
        };
    }

    private getInitialGuests(): Pick<
        IBookFormValues,
        EFormGroup.ADULT | EFormGroup.CHILDREN
    > {
        const {
            offerInfoByToken: {offerInfo},
            orderInfo: {guestsInfo},
            userInfo,
        } = this.props;

        if (!offerInfo) {
            return {
                adult: [],
                children: [],
            };
        }

        const {searchParams} = offerInfo;

        if (guestsInfo?.guests?.length) {
            return getGuestsByOrderInfo(guestsInfo, searchParams);
        }

        const {adults, childrenAges} = searchParams;
        const storageGuests = getBookingPassengers<IStorageBookGuest>(
            EFormKey.HOTEL_BOOK,
            userInfo,
        );
        const storageChildren = storageGuests.filter(
            guests => guests.ageGroup === 'children',
        );

        return {
            adult: storageGuests
                .filter(guests => guests.ageGroup === 'adult')
                .splice(0, adults)
                .map((adult, i) => ({
                    ...adult,
                    [EBookGuestFieldName.INDEX]: i,
                })),
            children: childrenAges.map((age, index) => ({
                ...storageChildren[index],
                age,
            })),
        };
    }

    private getInitialPriceInfo(): IBookPriceInfo {
        return this.props.bookUserSelect;
    }

    private getInitalFormValues(): IBookFormValues {
        const guests = this.getInitialGuests();
        const contacts = this.getInitialContacts();
        const priceInfo = this.getInitialPriceInfo();
        const whiteLabel = BookPage.getInitialWhiteLabel();

        return {
            ...guests,
            contacts,
            priceInfo,
            whiteLabel,
        };
    }

    private getTotalWithdrawPlusPoints = (): number | undefined => {
        const {promoCodesInfo, offerInfoByToken} = this.props;

        return getTotalWithdrawPlusPoints({
            promoCampaigns: offerInfoByToken?.offerInfo?.promoCampaigns,
            promoCodesInfo,
        });
    };

    private getTotalTopupPlusPoints = (): number => {
        const {promoCodesInfo, offerInfoByToken} = this.props;

        return getTotalTopupPlusPoints({
            promoCampaigns: offerInfoByToken?.offerInfo?.promoCampaigns,
            promoCodesInfo,
        });
    };

    private handleSubmit = (formValues: IBookFormValues): void => {
        const {
            offerInfoByToken: {offerInfo},
            deviceType,
            createOrder,
            userInfo,
        } = this.props;

        if (!offerInfo) {
            return;
        }

        this.setState({
            canRenderGoToForm: false,
            wasSessionKeyRenewed: false,
        });

        const {offerOrderInfo, priceInfo} = offerInfo;
        const finishPaymentPagePath = getFinishPaymentPagePath();
        const totalPrice = priceInfo?.hotelCharges.totals?.totalPrice;
        const plusMode =
            formValues[EFormGroup.PRICE_INFO]?.plusMode ||
            EYandexPlusApplicationMode.TOPUP;

        if (totalPrice) {
            reachGoal(EHotelsGoal.BOOK_PAY_BUTTON, {
                hotels: {payPrice: totalPrice},
            });
        }

        const hasPlus = userInfo.plusInfo?.hasPlus;

        if (hasPlus) {
            params({common: {plusMode}});
        }

        createOrder({
            offerOrderInfo,
            deviceType,
            finishPaymentPagePath,
            formValues: omit(formValues, [EFormGroup.WHITE_LABEL]),
            appliedPromoCampaigns: this.getAppliedPromoCampaigns(formValues),
        });

        sendPaymentCounter(SOLOMON_HOTELS_PAY_BUTTON_CLICK);
    };

    /* Helpers */

    private getAppliedPromoCampaigns = (
        formValues: IBookFormValues,
    ): IAppliedPromoCampaigns | undefined => {
        const isWhiteLabel = Boolean(this.props.whiteLabelConfig);

        if (isWhiteLabel) {
            return {
                whiteLabel: formValues[EFormGroup.WHITE_LABEL],
            };
        }

        const plusMode =
            formValues[EFormGroup.PRICE_INFO]?.plusMode ||
            EYandexPlusApplicationMode.TOPUP;

        const finalPoints =
            plusMode === EYandexPlusApplicationMode.WITHDRAW
                ? this.getTotalWithdrawPlusPoints()
                : this.getTotalTopupPlusPoints();

        return finalPoints
            ? {
                  yandexPlus: {
                      points: finalPoints,
                      mode: plusMode,
                  },
              }
            : undefined;
    };

    private getBookStatusQaAttributes(): IWithQaAttributes {
        const {
            offerInfoByToken: {isLoading, isSuccess},
        } = this.props;

        let qaStatus;

        switch (true) {
            case isLoading: {
                qaStatus = 'isOfferPending';

                break;
            }

            case isSuccess: {
                qaStatus = 'isOfferFetched';

                break;
            }

            default: {
                qaStatus = 'isOfferError';
            }
        }

        return prepareQaAttributes({
            parent: HOTELS_BOOK_PAGE_QA,
            current: qaStatus,
        });
    }

    private getRedirectBookOptions(): {
        path: string;
        statusCode: ERedirectStatusCodes;
    } | null {
        if (__CLIENT__) {
            return null;
        }

        const {location} = this.props;
        const {orderId, token} = getQueryByLocation(location);

        if (!orderId && !token) {
            return {
                path: internalUrl(URLs[EProjectName.HOTELS]),
                statusCode: ERedirectStatusCodes.TEMPORARILY,
            };
        }

        return null;
    }

    /* Render */

    private renderBookSteps(): ReactNode {
        const {
            deviceType,
            bookUserSelect,
            offerInfoByToken: {offerInfo},
        } = this.props;

        const isNoPrepayment =
            bookUserSelect.useDeferredPayments &&
            offerInfo?.deferredPaymentSchedule?.zeroFirstPayment;

        return (
            offerInfo && (
                <BookHeader
                    hotelInfo={offerInfo.hotelInfo}
                    searchParams={offerInfo.searchParams}
                    currentStep={BOOK_STEPS[0]}
                    deviceType={deviceType}
                    isNoPrepayment={isNoPrepayment}
                    {...prepareQaAttributes(HOTELS_BOOK_PAGE_QA)}
                />
            )
        );
    }

    private renderPageLoader(): ReactNode {
        const {
            location,
            orderInfo: {status},
            offerInfoByToken: {isLoading},
        } = this.props;

        const {orderId} = getQueryByLocation(location);
        const canRenderPageLoader =
            isLoading || (orderId && status === INITIAL);

        return (
            <BookLoader
                className={cx('loader')}
                title={i18nBlock.offerLoadingDotTitle()}
                description={i18nBlock.offerLoadingDotDescription()}
                isLoading={Boolean(canRenderPageLoader)}
            />
        );
    }

    private renderContent(): ReactNode {
        const {
            deviceType,
            offerInfoByToken: {offerInfo},
            orderInfo,
            userInfo,
            bookUserSelect,
        } = this.props;
        const {canRenderGoToForm, wasSessionKeyRenewed} = this.state;

        if (!offerInfo) {
            return null;
        }

        const {allGuestsRequired} = offerInfo;

        const initialValues = this.getInitalFormValues();

        return (
            <Form<IBookFormValues>
                onSubmit={this.handleSubmit}
                decorators={[validateOnSubmit]}
                mutators={{focusFirstInvalidField, setFieldData}}
                initialValues={initialValues}
                validationInfo={optionalAdultsValidationInfo(allGuestsRequired)}
                subscription={{}}
                render={({handleSubmit, form}): ReactNode => (
                    <form
                        name={EFormKey.HOTEL_BOOK}
                        onSubmit={handleSubmit}
                        autoComplete="off"
                        noValidate
                    >
                        <FormController form={form}>
                            {({
                                focusFirstField,
                                focusFieldByName,
                                changeFieldByName,
                                submitForm,
                                formState,
                            }): React.ReactElement => (
                                <BookPageContent
                                    offerInfo={offerInfo}
                                    orderInfo={orderInfo}
                                    bookUserSelect={bookUserSelect}
                                    tokenByLocation={this.getOfferToken()}
                                    canRenderGoToForm={canRenderGoToForm}
                                    wasSessionKeyRenewed={wasSessionKeyRenewed}
                                    deviceType={deviceType}
                                    userInfo={userInfo}
                                    focusFirstField={focusFirstField}
                                    focusFieldByName={focusFieldByName}
                                    submitForm={submitForm}
                                    formState={formState}
                                    changeFieldByName={changeFieldByName}
                                    {...prepareQaAttributes(
                                        HOTELS_BOOK_PAGE_QA,
                                    )}
                                />
                            )}
                        </FormController>
                    </form>
                )}
            />
        );
    }

    renderMessenger(): ReactNode {
        return (
            <HangingYandexMessenger
                entrypoint="hotelBookPage"
                metrikaGoal={EHotelsGoal.HOTELS_CHAT_CLICK}
            />
        );
    }

    render(): ReactNode {
        const {
            deviceType,
            offerInfoByToken: {isSuccess},
        } = this.props;

        const redirectOptions = this.getRedirectBookOptions();

        if (redirectOptions) {
            return (
                <RedirectWithStatus
                    to={redirectOptions.path}
                    statusCode={redirectOptions.statusCode}
                />
            );
        }

        return (
            <HotelsBookLayout deviceType={deviceType} hasLoader={!isSuccess}>
                <div {...this.getBookStatusQaAttributes()}>
                    <BookRedirectController />
                    {isSuccess && this.renderBookSteps()}
                    {this.renderPageLoader()}
                    {isSuccess && this.renderContent()}
                    {isSuccess && this.renderMessenger()}
                </div>
            </HotelsBookLayout>
        );
    }
}

export default BookPage;
