import {PureComponent, ReactNode} from 'react';
import _times from 'lodash/times';

import {IWithDeviceType} from 'types/withDeviceType';
import {EFormGroup} from 'types/hotels/book/IBookFormValues';
import {IBookSearchParams} from 'server/api/HotelsBookAPI/types/IBookOffer';
import {IWithWhiteLabelConfig} from 'types/common/whiteLabel/IWithWhiteLabelConfig';

import {
    prepareQaAttributes,
    IWithQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import validationInfo from '../../utilities/validationInfo';
import {excludeAdultGuests} from 'projects/hotels/pages/BookPage/utilities/excludeAdultGuests';

import * as i18nBlock from 'i18n/hotels-BookGuestsForm';

import Form from 'components/Form/Form';
import Separator from 'components/Separator/Separator';
import Intersperse from 'components/Intersperse/Intersperse';
import BookingLayout from 'components/Layouts/BookingLayout/BookingLayout';
import BookingContactsForm from 'components/BookingContactsForm/BookingContactsForm';
import BookingPassengerForm from 'components/BookingPassengerForm/BookingPassengerForm';
import BookingWhiteLabelForm from 'components/BookingWhiteLabelForm/BookingWhiteLabelForm';
import WhiteLabelBookingCard from './components/WhiteLabelBookingCard/WhiteLabelBookingCard';

import {TBookPageContainerProps} from './BookGuestsFormContainer';

import cx from './BookGuestsForm.scss';

const BOOK_FORM_QA = 'book-form';

export interface IBookGuestsFormOwnProps
    extends IWithDeviceType,
        IWithQaAttributes,
        IWithWhiteLabelConfig {
    searchParams?: IBookSearchParams;
    allGuestsRequired: boolean;
}

interface IBookGuestsFormProps
    extends TBookPageContainerProps,
        IBookGuestsFormOwnProps {}

class BookGuestsForm extends PureComponent<IBookGuestsFormProps> {
    /* Render field */

    private renderGuestForm(
        title: string,
        index: number,
        isChild: boolean,
    ): ReactNode {
        const {deviceType, passengers, searchParams, allGuestsRequired} =
            this.props;
        const {adults = 0, childrenAges = []} = searchParams || {};
        const totalPassengersLength = adults + childrenAges.length;
        const suitablePassengers = isChild
            ? excludeAdultGuests(passengers.data, searchParams?.startDate)
            : passengers.data;

        return (
            <div
                className={cx('passengerForm')}
                style={{zIndex: totalPassengersLength - index + 1}}
            >
                <BookingPassengerForm
                    title={i18nBlock.guestDotTitle({
                        index: index + 1,
                        ageGroup: title,
                        first: index === 0,
                    })}
                    passengers={suitablePassengers}
                    validationInfo={validationInfo(allGuestsRequired)}
                    deviceType={deviceType}
                    isOnlyName
                    {...prepareQaAttributes({
                        parent: this.props,
                        current: `${BOOK_FORM_QA}-guests`,
                        key: index,
                    })}
                />
            </div>
        );
    }

    /* Render fields */

    private renderGuestsFields(): ReactNode {
        const {searchParams} = this.props;
        const {adults = 0, childrenAges = []} = searchParams || {};

        return (
            <Intersperse separator={<Separator />}>
                {_times(adults, index => (
                    <Form.FieldGroup groupId={EFormGroup.ADULT} key={index}>
                        <Form.FieldGroup groupId={index}>
                            <BookingLayout.Card.Section>
                                {this.renderGuestForm(
                                    i18nBlock.adult(),
                                    index,
                                    false,
                                )}
                            </BookingLayout.Card.Section>
                        </Form.FieldGroup>
                    </Form.FieldGroup>
                ))}

                {childrenAges.map((age, index) => (
                    <Form.FieldGroup key={index} groupId={EFormGroup.CHILDREN}>
                        <Form.FieldGroup groupId={index}>
                            <BookingLayout.Card.Section>
                                {this.renderGuestForm(
                                    i18nBlock.child({age}),
                                    adults + index,
                                    true,
                                )}
                            </BookingLayout.Card.Section>
                        </Form.FieldGroup>
                    </Form.FieldGroup>
                ))}
            </Intersperse>
        );
    }

    renderEmailAndPhoneFields(): ReactNode {
        return (
            <Form.FieldGroup groupId={EFormGroup.CONTACTS}>
                <BookingContactsForm
                    {...prepareQaAttributes({
                        parent: this.props,
                        current: `${BOOK_FORM_QA}-contacts`,
                    })}
                />
            </Form.FieldGroup>
        );
    }

    renderCustomerNumberField(): ReactNode {
        return (
            <Form.FieldGroup groupId={EFormGroup.WHITE_LABEL}>
                <BookingWhiteLabelForm />
            </Form.FieldGroup>
        );
    }

    render(): ReactNode {
        const {whiteLabelConfig} = this.props;
        const isWhiteLabel = Boolean(whiteLabelConfig);

        return (
            <div
                {...prepareQaAttributes({
                    parent: this.props,
                    current: BOOK_FORM_QA,
                })}
            >
                <BookingLayout.Forms>
                    <BookingLayout.Card sectioned>
                        {this.renderGuestsFields()}
                    </BookingLayout.Card>
                    {isWhiteLabel && (
                        <WhiteLabelBookingCard>
                            {this.renderCustomerNumberField()}
                        </WhiteLabelBookingCard>
                    )}
                    <BookingLayout.Card className={cx('contacts')}>
                        {this.renderEmailAndPhoneFields()}
                    </BookingLayout.Card>
                </BookingLayout.Forms>
            </div>
        );
    }
}

export default BookGuestsForm;
