import React, {
    FunctionComponent,
    useCallback,
    useEffect,
    useState,
} from 'react';
import {FormApi} from 'final-form';

import {IBookFormValues} from 'types/hotels/book/IBookFormValues';

export type TFormEvent = () => void;
export type TFocusFieldEvent = (name: string) => void;
export type TChangeFieldEvent = (
    name: string,
    value?: string | number | boolean | null,
) => void;

interface IRenderFormControllerProps {
    focusFirstField: TFormEvent;
    focusFieldByName: TFocusFieldEvent;
    changeFieldByName: TChangeFieldEvent;
    submitForm: TFormEvent;
    formState: IBookFormValues;
}

interface IFormControllerProps {
    form: FormApi<IBookFormValues>;
    children: (props: IRenderFormControllerProps) => React.ReactElement;
}

const FormController: FunctionComponent<IFormControllerProps> = ({
    form,
    children,
}) => {
    const [formState, setFormState] = useState<IBookFormValues>(
        form.getState().values,
    );

    useEffect(() => {
        return form.subscribe(
            ({values}) => {
                setFormState(values);
            },
            {values: true},
        );
    }, [form]);

    const focusFirstField = useCallback(() => {
        const [fieldName] = form.getRegisteredFields();

        form.focus(fieldName as any);
    }, [form]);

    const focusFieldByName = useCallback(
        (name: string) => {
            form.focus(name as any);
        },
        [form],
    );

    const changeFieldByName = useCallback<TChangeFieldEvent>(
        (name, value) => {
            form.change(name as any, value);
        },
        [form],
    );

    const submitForm = useCallback(() => {
        form.submit();
    }, [form]);

    return children({
        focusFirstField,
        focusFieldByName,
        changeFieldByName,
        submitForm,
        formState,
    });
};

export default FormController;
