import React, {useState, useCallback, useEffect, useMemo} from 'react';

import {
    IAppliedPromoCodesInfo,
    IPromoCodeApplicationResult,
} from 'types/hotels/book/IApplyPromoCodes';

import {prepareQaAttributes} from 'utilities/qaAttributes/qaAttributes';
import IPrice from 'utilities/currency/PriceInterface';
import {CurrencyType} from 'utilities/currency/CurrencyType';

import * as i18nBlock from 'i18n/hotels-BookPriceInfo';

import PromoCodeError from 'projects/hotels/components/PromoCodeError/PromoCodeError';
import PriceErrorModal from 'components/PriceErrorModal/PriceErrorModal';

const MODAL_QA = 'reservedWithRestrictionsModal';

interface IReservedWithRestrictionsModalProps {
    promoCodesInfo?: IAppliedPromoCodesInfo;
    codeApplicationResults?: IPromoCodeApplicationResult;
    isVisible: boolean;
    onContinueClick: () => void;
    onCancelClick: () => void;
}

const ReservedWithRestrictionsModal: React.FC<IReservedWithRestrictionsModalProps> =
    ({
        promoCodesInfo,
        codeApplicationResults,
        onContinueClick,
        onCancelClick,
        isVisible,
    }) => {
        const [isDisabled, setDisabled] = useState(false);

        const handleCancelClick = useCallback(() => {
            onCancelClick();
            setDisabled(true);
        }, [onCancelClick, setDisabled]);

        const handleContinueClick = useCallback(() => {
            onContinueClick();
            setDisabled(true);
        }, [onContinueClick, setDisabled]);

        useEffect(() => {
            if (isVisible) {
                setDisabled(false);
            }
        }, [isVisible]);

        const errorText = useMemo(() => {
            return (
                <>
                    <PromoCodeError status={codeApplicationResults?.type} />
                    {'. '}
                    {i18nBlock.promocodeDotCancellationDescription()}
                </>
            );
        }, [codeApplicationResults]);

        // Хак для того чтобы правильно показать изменение стоимости при неподходящем промокоде, для бэка проблема тяжелая, поэтому пока хакнули на фронте,
        // фикстаск: https://st.yandex-team.ru/HOTELS-5518
        const promocodeDiscount =
            promoCodesInfo?.codeApplicationResults?.[0]?.discountAmount;

        const canHackDiscount =
            typeof promocodeDiscount?.value === 'number' &&
            typeof promoCodesInfo?.discountedAmount?.value === 'number' &&
            promocodeDiscount?.currency ===
                promoCodesInfo?.discountedAmount?.currency;

        const hackedPrice: IPrice | undefined = canHackDiscount
            ? {
                  value:
                      (promoCodesInfo?.discountedAmount?.value || 0) +
                      (promocodeDiscount?.value || 0),
                  currency:
                      promoCodesInfo?.discountedAmount?.currency ||
                      promocodeDiscount?.currency ||
                      CurrencyType.RUB,
              }
            : undefined;

        return (
            <PriceErrorModal
                title={i18nBlock.promocodeDotCancelled()}
                text={errorText}
                price={hackedPrice || promoCodesInfo?.originalAmount}
                lastPrice={promoCodesInfo?.discountedAmount}
                cancelButtonText={i18nBlock.promocodeDotErrorModalDotCancelButton()}
                isVisible={isVisible}
                isDisabled={isDisabled}
                onCancelClick={handleCancelClick}
                onRetryClick={handleContinueClick}
                {...prepareQaAttributes(MODAL_QA)}
            />
        );
    };

export default ReservedWithRestrictionsModal;
