import memoize from 'lodash/memoize';

import {EFormKey} from 'constants/form/EFormKey';
import {
    FREE_PHONE_REGEXP,
    getEmailValidationRegExp,
} from 'constants/form/validationRules';
import {EFieldName} from 'components/BookingPassengerForm/constants/fieldNames';
import {EFieldName as EContactsFieldName} from 'components/BookingContactsForm/constants/fieldNames';

import {
    IFormValidationInfo,
    IFieldValidation,
} from 'types/common/validation/form';
import {
    EValidationType,
    IDependencyCondition,
    IDependentValidation,
} from 'types/common/validation/validation';
import {
    EBookGuestFieldName,
    EFormGroup,
} from 'types/hotels/book/IBookFormValues';

import * as i18nBookGuestsFormBlock from 'i18n/hotels-BookGuestsForm';

const NAME_MAX_LENGTH = 80;
const EMAIL_MAX_LENGTH = 40;
const PHONE_MAX_LENGTH = 80;

const FIRST_GUEST_INDEX = 0;

const nameValidation: IFieldValidation = {
    blur: [
        {
            type: EValidationType.regex,
            params: '^[a-zA-Zа-яА-ЯёЁ-\\s]*$',
            errorMessage:
                i18nBookGuestsFormBlock.validateDotGuestNameExtended(),
        },
        {
            type: EValidationType.maxLength,
            params: NAME_MAX_LENGTH,
            errorMessage: i18nBookGuestsFormBlock.validateDotMaxLengthExceeded({
                maxLength: NAME_MAX_LENGTH,
            }),
        },
    ],
};

const requiredOnSubmit: IFieldValidation = {
    submit: [
        {
            type: EValidationType.required,
            params: true,
            errorMessage: i18nBookGuestsFormBlock.validateDotRequiredField(),
        },
    ],
};

const requiredAdultsValidation = (
    allGuestsRequired: boolean,
): IDependentValidation => {
    const conditions: IDependencyCondition[] = allGuestsRequired
        ? []
        : [
              {
                  path: {
                      fieldName: EBookGuestFieldName.INDEX,
                  },
                  value: [
                      {
                          type: EValidationType.oneOf,
                          params: [FIRST_GUEST_INDEX],
                      },
                  ],
              },
          ];

    return {
        conditions,
        validation: requiredOnSubmit,
    };
};

// Если заполнена фамилия то нужно заполнить и имя
const optionalFirstNameValidation: IDependentValidation = {
    conditions: [
        {
            path: {
                fieldName: EFieldName.lastName,
            },
            value: [
                {
                    type: EValidationType.required,
                    params: true,
                },
            ],
        },
    ],
    validation: requiredOnSubmit,
};

const requiredChildrenValidation = (
    allGuestsRequired: boolean,
): IDependentValidation => ({
    conditions: [],
    validation: allGuestsRequired ? requiredOnSubmit : {},
});

// Если заполнено имя то нужно заполнить и фамилию
const optionalLastNameValidation: IDependentValidation = {
    conditions: [
        {
            path: {
                fieldName: EFieldName.firstName,
            },
            value: [
                {
                    type: EValidationType.required,
                    params: true,
                },
            ],
        },
    ],
    validation: requiredOnSubmit,
};

const validationInfo = (allGuestsRequired: boolean): IFormValidationInfo => ({
    id: EFormKey.HOTEL_BOOK,

    fieldGroups: [
        {
            id: EFormGroup.ADULT,
            fields: [
                {
                    name: EFieldName.firstName,
                    validation: nameValidation,
                    dependentValidations: [
                        requiredAdultsValidation(allGuestsRequired),
                        optionalFirstNameValidation,
                    ],
                },
                {
                    name: EFieldName.lastName,
                    validation: nameValidation,
                    dependentValidations: [
                        requiredAdultsValidation(allGuestsRequired),
                        optionalLastNameValidation,
                    ],
                },
            ],
        },
        {
            id: EFormGroup.CHILDREN,
            fields: [
                {
                    name: EFieldName.firstName,
                    validation: nameValidation,
                    dependentValidations: [
                        requiredChildrenValidation(allGuestsRequired),
                        optionalFirstNameValidation,
                    ],
                },
                {
                    name: EFieldName.lastName,
                    validation: nameValidation,
                    dependentValidations: [
                        requiredChildrenValidation(allGuestsRequired),
                        optionalLastNameValidation,
                    ],
                },
            ],
        },
        {
            id: EFormGroup.CONTACTS,
            fields: [
                {
                    name: EContactsFieldName.email,
                    validation: {
                        blur: [
                            {
                                type: EValidationType.regex,
                                params: getEmailValidationRegExp({
                                    allowRussianCharacters: true,
                                }),
                                errorMessage:
                                    i18nBookGuestsFormBlock.validateDotEmail(),
                            },
                            {
                                type: EValidationType.maxLength,
                                params: EMAIL_MAX_LENGTH,
                                errorMessage:
                                    i18nBookGuestsFormBlock.validateDotMaxLengthExceeded(
                                        {maxLength: EMAIL_MAX_LENGTH},
                                    ),
                            },
                        ],
                        submit: [
                            {
                                type: EValidationType.required,
                                params: true,
                                errorMessage:
                                    i18nBookGuestsFormBlock.validateDotRequiredField(),
                            },
                        ],
                    },
                },
                {
                    name: EContactsFieldName.phone,
                    validation: {
                        blur: [
                            {
                                type: EValidationType.regex,
                                params: FREE_PHONE_REGEXP,
                                errorMessage:
                                    i18nBookGuestsFormBlock.validateDotPhoneFormat(),
                            },
                            {
                                type: EValidationType.maxLength,
                                params: PHONE_MAX_LENGTH,
                                errorMessage:
                                    i18nBookGuestsFormBlock.validateDotMaxLengthExceeded(
                                        {maxLength: PHONE_MAX_LENGTH},
                                    ),
                            },
                        ],
                        submit: [
                            {
                                type: EValidationType.required,
                                params: true,
                                errorMessage:
                                    i18nBookGuestsFormBlock.validateDotRequiredField(),
                            },
                        ],
                    },
                },
            ],
        },
    ],
});

// Мемоизация нужна, чтобы не вызывать ререндер finalForm разными объектами
export default memoize(validationInfo);
