import {PureComponent, ReactNode} from 'react';

/* Constants */
import {EProjectName} from 'constants/common';
import {ERedirectStatusCodes} from 'constants/redirectStatusCodes';
import {URLs} from 'constants/urls';

import {EHotelsGoal} from 'utilities/metrika/types/goals/hotels';
import {
    EGeoRegionBlockType,
    IBreadcrumbsBlock,
    IHotelsBlock,
    IHotelsFiltersBlock,
    IRegionLinkSetBlock,
    ISearchFormBlock,
    TGeoRegionBlock,
} from 'types/hotels/geoRegion/IGeoRegionInfo';
import {
    ETextBlocksType,
    ISectionTextBlock,
    TLinkTextBlock,
} from 'types/common/seoPages/ITextBlocks';
import {EFooterProject} from 'components/Footer/types';
import {ICrossSaleHotelsBlock} from 'types/common/seoPages/ICrossSaleHotelsBlock';
import {ECommonLandingBlockType} from 'types/common/seoPages/ECommonLandingBlockType';

import {internalUrl} from 'utilities/url';
import {reachGoal} from 'utilities/metrika';
import {hotelsURLs} from 'projects/hotels/utilities/urls';
import {getRegionSlugByRouteMatch} from 'projects/hotels/utilities/getRegionIdentifier/getRegionIdentifier';
import {
    deviceModDesktop,
    deviceModMobile,
    deviceMods,
} from 'utilities/stylesUtils';
import scrollTo from 'utilities/dom/scrollTo';
import sendMetrikaExtraVisitAndUserParams from 'projects/hotels/utilities/metrika/sendMetrikaExtraVisitAndUserParams';
import {prepareQaAttributes} from 'utilities/qaAttributes/qaAttributes';

import * as i18nBlock from 'i18n/hotels-GeoRegionPage';

import LayoutDefault from 'components/Layouts/LayoutDefault/LayoutDefault';
import SearchFormBlock from './components/SearchFormBlock/SearchFormBlock';
import HotelsBlock from './components/HotelsBlock/HotelsBlock';
import SectionTextBlock from 'components/SectionTextBlock/SectionTextBlock';
import {TwoColumnLayout} from 'components/Layouts/TwoColumnLayout/TwoColumnLayout';
import RedirectWithStatus from 'components/RedirectWithStatus/RedirectWithStatus';
import GeoRegionMetaHelmet from './components/GeoRegionMetaHelmet/GeoRegionMetaHelmet';
import GeoRegionPageBreadCrumbs from 'projects/hotels/pages/GeoRegionPage/components/GeoRegionPageBreadCrumbs/GeoRegionPageBreadCrumbs';
import RegionCrossLinks from './components/RegionCrossLinks/RegionCrossLinks';
import Filters from 'projects/hotels/pages/GeoRegionPage/components/Filters/Filters';
import CrossSaleHotelsBlock from 'components/CrossSaleHotelsBlock/CrossSaleHotelsBlock';

import HotelsSearchInformationProvider from 'projects/hotels/containers/HotelsSearchInformationProvider/HotelsSearchInformationProvider';

import WhiteLabelConfigContext from 'contexts/WhiteLabelConfigContext';

import {TGeoRegionPagePropsContainer} from './GeoRegionPageContainer';

import cx from './GeoRegionPage.scss';

interface IGeoRegionPageProps extends TGeoRegionPagePropsContainer {}

const RIGHT_COLUMN_OFFSET = 10;
const RIGHT_COLUMN_WIDTH = 80;
const ROOT_QA = 'hotels-geoPage';

class GeoRegionPage extends PureComponent<IGeoRegionPageProps> {
    reachedMetrikaParams: boolean = false;

    renderAdfox = true;

    componentDidMount(): void {
        const {fetchGeoRegionInfo, match, geoRegionInfo} = this.props;

        if (!geoRegionInfo?.data && match.params.regionSlug) {
            fetchGeoRegionInfo({
                regionSlug: match.params.regionSlug,
                filterSlug: match.params.filterSlug,
            });
        }

        reachGoal(EHotelsGoal.REGION_PAGE_VISITED, {
            hotels: {
                didVisitSeoRegionPage: 1,
            },
        });

        this.reachMetrikaParams();
    }

    componentDidUpdate(prevProps: Readonly<IGeoRegionPageProps>): void {
        const {fetchGeoRegionInfo, match} = this.props;

        if (
            prevProps.location.pathname !== location.pathname &&
            match.params.regionSlug
        ) {
            scrollTo({top: 0});
            fetchGeoRegionInfo({
                regionSlug: match.params.regionSlug,
                filterSlug: match.params.filterSlug,
            });
        }

        this.reachMetrikaParams();
    }

    /* Helpers */

    reachMetrikaParams(): void {
        if (!this.reachedMetrikaParams && this.props.geoRegionInfo.data) {
            const {extraVisitAndUserParams} = this.props.geoRegionInfo.data;

            sendMetrikaExtraVisitAndUserParams(extraVisitAndUserParams);

            this.reachedMetrikaParams = true;
        }
    }

    getRedirectGeoPageOptions(): {
        path: string;
        statusCode: ERedirectStatusCodes;
    } | null {
        const {match} = this.props;
        const blocks = this.props.geoRegionInfo?.data?.blocks || [];
        const searchFormBlock = this.getSearchFormBlock(blocks);
        const regionSlugByResponse =
            searchFormBlock?.data?.searchFormParams?.regionSlug;
        const regionSlugByRouteMatch = getRegionSlugByRouteMatch(match);

        if (__CLIENT__) {
            return null;
        }

        if (!regionSlugByResponse) {
            return {
                path: internalUrl(URLs.notFound),
                statusCode: ERedirectStatusCodes.TEMPORARILY,
            };
        }

        if (
            regionSlugByResponse &&
            regionSlugByResponse !== regionSlugByRouteMatch
        ) {
            return {
                path: hotelsURLs.getRegionUrl({
                    regionSlug: regionSlugByResponse,
                    filterSlug: match.params.filterSlug,
                }),
                statusCode: ERedirectStatusCodes.PERMANENTLY,
            };
        }

        return null;
    }

    private getSearchFormBlock = (
        blocks: TGeoRegionBlock[],
    ): ISearchFormBlock => {
        const [searchFormBlock] = blocks.filter(
            ({type}) => type === EGeoRegionBlockType.SEARCH_FORM_BLOCK,
        );

        return searchFormBlock as ISearchFormBlock;
    };

    private getHotelsFiltersBlock = (
        blocks: TGeoRegionBlock[],
    ): IHotelsFiltersBlock => {
        const [searchFormBlock] = blocks.filter(
            ({type}) => type === EGeoRegionBlockType.HOTELS_FILTERS_BLOCK,
        );

        return searchFormBlock as IHotelsFiltersBlock;
    };

    private getHotelBlocks = (blocks: TGeoRegionBlock[]): IHotelsBlock[] =>
        blocks.filter(
            ({type}) => type === EGeoRegionBlockType.HOTELS_BLOCK,
        ) as IHotelsBlock[];

    private getTextBlocks = (blocks: TGeoRegionBlock[]): ISectionTextBlock[] =>
        blocks.filter(
            ({type}) => type === ETextBlocksType.sectionTextBlock,
        ) as ISectionTextBlock[];

    private getRegionLinksBlocks = (
        blocks: TGeoRegionBlock[],
    ): IRegionLinkSetBlock[] =>
        blocks.filter(
            ({type}) => type === EGeoRegionBlockType.REGION_LINK_SET_BLOCK,
        ) as IRegionLinkSetBlock[];

    private getHotelsCrossSaleBlocks = (
        blocks: TGeoRegionBlock[],
    ): ICrossSaleHotelsBlock[] =>
        blocks.filter(
            ({type}) => type === ECommonLandingBlockType.HOTELS_CROSS_SALE,
        ) as ICrossSaleHotelsBlock[];

    private getBreadcrumbsBlock = (
        blocks: TGeoRegionBlock[],
    ): IBreadcrumbsBlock =>
        blocks.find(
            ({type}) => type === EGeoRegionBlockType.BREADCRUMBS_BLOCK,
        ) as IBreadcrumbsBlock;

    private getBlocksWithoutSearchForm = (
        blocks: TGeoRegionBlock[],
    ): Exclude<TGeoRegionBlock, ISearchFormBlock>[] =>
        blocks.filter(
            block => block.type !== EGeoRegionBlockType.SEARCH_FORM_BLOCK,
        ) as Exclude<TGeoRegionBlock, ISearchFormBlock>[];

    private getBlocksWithFilterAtRightPosition = (
        blocks: TGeoRegionBlock[],
    ): TGeoRegionBlock[] => {
        const filters = blocks.find(
            ({type}) => type === EGeoRegionBlockType.HOTELS_FILTERS_BLOCK,
        );
        const insertIndex = blocks.findIndex(
            ({type}) => type === EGeoRegionBlockType.HOTELS_BLOCK,
        );

        if (!filters || insertIndex === -1) {
            return blocks;
        }

        const filteredBlocks = blocks.filter(
            ({type}) => type !== EGeoRegionBlockType.HOTELS_FILTERS_BLOCK,
        );

        return [
            ...filteredBlocks.slice(0, insertIndex),
            filters,
            ...filteredBlocks.slice(insertIndex),
        ];
    };

    onLinkBlockClick = (linkBlock: TLinkTextBlock): void => {
        if (linkBlock.type === ETextBlocksType.hotelLinkBlock) {
            reachGoal(EHotelsGoal.REGION_PAGE_FAQ_HOTEL_CLICK);
        }
    };

    /* Render */

    renderGeoRegionMetaHelmet(): ReactNode {
        const {geoRegionInfo, schemaNonce, match} = this.props;
        const seoInfo = geoRegionInfo?.data?.seoInfo;
        const regionSlugByRouteMatch = getRegionSlugByRouteMatch(match);

        if (seoInfo && regionSlugByRouteMatch) {
            const absoluteRegionUrl = hotelsURLs.getAbsoluteRegionUrl({
                regionSlug: regionSlugByRouteMatch,
                filterSlug: match.params.filterSlug,
            });

            return (
                <GeoRegionMetaHelmet
                    absoluteRegionUrl={absoluteRegionUrl}
                    seoInfo={seoInfo}
                    schemaNonce={schemaNonce}
                />
            );
        }

        return null;
    }

    renderSearchFormBlock(block: ISearchFormBlock, index: number): ReactNode {
        const {deviceType} = this.props;

        return (
            <SearchFormBlock
                key={`${block.type}${index}`}
                className={cx(deviceMods('searchFormBlock', deviceType))}
                deviceType={deviceType}
                block={block}
            />
        );
    }

    renderHotelsBlock(
        block: IHotelsBlock,
        renderAdfox: boolean,
        index: number,
    ): ReactNode {
        const {deviceType} = this.props;

        return (
            <HotelsBlock
                key={`${block.type}${index}`}
                className={cx(deviceMods('hotelsBlock', deviceType))}
                deviceType={deviceType}
                block={block}
                renderAdfox={renderAdfox}
                {...prepareQaAttributes(ROOT_QA)}
            />
        );
    }

    renderSectionTextBlock(block: ISectionTextBlock, index: number): ReactNode {
        const {deviceType} = this.props;

        return (
            <SectionTextBlock
                key={`${block.type}${index}`}
                className={cx('sectionTextBlock')}
                deviceType={deviceType}
                block={block}
                onLinkClick={this.onLinkBlockClick}
            />
        );
    }

    renderRegionLinkSetBlock = (
        block: IRegionLinkSetBlock,
        index: number,
    ): ReactNode => {
        if (
            !block.data.subsets ||
            block.data.subsets.every(s => s.data.links.length === 0)
        ) {
            return null;
        }

        const {deviceType} = this.props;

        return (
            <RegionCrossLinks
                key={`${block.type}${index}`}
                className={cx(deviceMods('regionCrossLinks', deviceType))}
                block={block}
            />
        );
    };

    private renderHotelsCrossSaleBlock = (
        block: ICrossSaleHotelsBlock,
        index: number,
    ): ReactNode => {
        const {deviceType} = this.props;

        return (
            <CrossSaleHotelsBlock
                {...block.data}
                key={`${block.type}${index}`}
                vertical={EProjectName.HOTELS}
                className={cx(deviceMods('hotelsCrossSaleBlock', deviceType))}
                linkType="search"
            />
        );
    };

    renderGeoBlock = (block: TGeoRegionBlock, index: number): ReactNode => {
        switch (block.type) {
            case EGeoRegionBlockType.SEARCH_FORM_BLOCK: {
                return this.renderSearchFormBlock(block, index);
            }

            case EGeoRegionBlockType.HOTELS_FILTERS_BLOCK: {
                return this.renderHotelsFiltersBlock();
            }

            case EGeoRegionBlockType.HOTELS_BLOCK: {
                // Only render adfox for the first encountered hotelsBlock
                const renderedBlock = this.renderHotelsBlock(
                    block,
                    this.renderAdfox,
                    index,
                );

                if (this.renderAdfox) {
                    this.renderAdfox = false;
                }

                return renderedBlock;
            }

            case EGeoRegionBlockType.REGION_LINK_SET_BLOCK: {
                return this.renderRegionLinkSetBlock(block, index);
            }

            case ETextBlocksType.sectionTextBlock: {
                return this.renderSectionTextBlock(block, index);
            }

            case ECommonLandingBlockType.HOTELS_CROSS_SALE: {
                return this.renderHotelsCrossSaleBlock(block, index);
            }

            default: {
                return null;
            }
        }
    };

    renderDesktopContent(): ReactNode {
        const {deviceType} = this.props;
        const blocks = this.props.geoRegionInfo?.data?.blocks;

        if (blocks?.length) {
            const hotelBlocks = this.getHotelBlocks(blocks);
            const textBlocks = this.getTextBlocks(blocks);
            const regionLinksBlocks = this.getRegionLinksBlocks(blocks);
            const hotelsCrossSaleBlocks = this.getHotelsCrossSaleBlocks(blocks);

            return (
                <WhiteLabelConfigContext.Consumer>
                    {(config): ReactNode => {
                        return config ? (
                            <div
                                className={cx(
                                    'content',
                                    deviceModDesktop('content', deviceType),
                                )}
                            >
                                {hotelsCrossSaleBlocks.map(
                                    this.renderHotelsCrossSaleBlock,
                                )}
                                {hotelBlocks.map(this.renderGeoBlock)}
                                {regionLinksBlocks.map(
                                    this.renderRegionLinkSetBlock,
                                )}
                            </div>
                        ) : (
                            <TwoColumnLayout
                                className={cx(
                                    'content',
                                    deviceModDesktop('content', deviceType),
                                )}
                                tagName="section"
                                deviceType={deviceType}
                                rightColumnOffset={RIGHT_COLUMN_OFFSET}
                                rightColumnWidth={RIGHT_COLUMN_WIDTH}
                            >
                                <TwoColumnLayout.LeftColumn>
                                    {hotelsCrossSaleBlocks.map(
                                        this.renderHotelsCrossSaleBlock,
                                    )}
                                    {hotelBlocks.map(this.renderGeoBlock)}
                                    {regionLinksBlocks.map(
                                        this.renderRegionLinkSetBlock,
                                    )}
                                </TwoColumnLayout.LeftColumn>
                                <TwoColumnLayout.RightColumn>
                                    {textBlocks.map(this.renderGeoBlock)}
                                </TwoColumnLayout.RightColumn>
                            </TwoColumnLayout>
                        );
                    }}
                </WhiteLabelConfigContext.Consumer>
            );
        }

        return null;
    }

    renderMobileContent(): ReactNode {
        const {deviceType, geoRegionInfo} = this.props;
        const blocks = geoRegionInfo?.data?.blocks;

        if (blocks?.length) {
            const blocksWithoutSearchForm =
                this.getBlocksWithoutSearchForm(blocks);

            return (
                <section
                    className={cx(
                        'content',
                        deviceModMobile('content', deviceType),
                    )}
                >
                    {this.getBlocksWithFilterAtRightPosition(
                        blocksWithoutSearchForm,
                    ).map(this.renderGeoBlock)}
                </section>
            );
        }

        return null;
    }

    renderFooterDisclaimer(): ReactNode {
        const {deviceType} = this.props;

        return (
            <div
                className={cx(
                    'footerDisclaimer',
                    deviceMods('footerDisclaimer', deviceType),
                )}
            >
                {i18nBlock.footerDisclaimer()}
            </div>
        );
    }

    renderHotelsFiltersBlock(): ReactNode {
        const {deviceType, resetFilter} = this.props;

        if (deviceType.isDesktop) {
            return null;
        }

        const blocks = this.props.geoRegionInfo?.data?.blocks;
        const hotelsFiltersBlock = this.getHotelsFiltersBlock(blocks || []);

        return hotelsFiltersBlock ? (
            <Filters
                className={cx('filters')}
                filters={hotelsFiltersBlock.data.filterInfo}
                searchParams={hotelsFiltersBlock.data.searchParams}
                onResetFilter={resetFilter}
            />
        ) : null;
    }

    renderBreadcrumbs(): ReactNode {
        const {deviceType} = this.props;
        const blocks = this.props.geoRegionInfo?.data?.blocks;

        if (blocks?.length) {
            const breadcrumbsBlock = this.getBreadcrumbsBlock(blocks);

            if (!breadcrumbsBlock) {
                return null;
            }

            return (
                <GeoRegionPageBreadCrumbs
                    className={cx(
                        'breadCrumbs',
                        deviceMods('breadCrumbs', deviceType),
                    )}
                    breadcrumbs={breadcrumbsBlock.data.breadcrumbs}
                    {...prepareQaAttributes({
                        parent: ROOT_QA,
                        current: 'breadcrumps',
                    })}
                />
            );
        }

        return null;
    }

    render(): ReactNode {
        const {deviceType} = this.props;
        const {isDesktop} = deviceType;
        const blocks = this.props.geoRegionInfo?.data?.blocks;
        const redirectOptions = this.getRedirectGeoPageOptions();

        this.renderAdfox = true;

        if (redirectOptions) {
            return (
                <RedirectWithStatus
                    to={redirectOptions.path}
                    statusCode={redirectOptions.statusCode}
                />
            );
        }

        if (blocks?.length) {
            const searchFormBlock = this.getSearchFormBlock(blocks);

            return (
                <LayoutDefault
                    hasSideSheetNavigation
                    showNavigation
                    project={EProjectName.HOTELS}
                    footerType={EFooterProject.HOTELS}
                    className={cx(deviceMods('geoPage', deviceType))}
                >
                    {this.renderGeoRegionMetaHelmet()}
                    {Boolean(searchFormBlock) &&
                        this.renderGeoBlock(searchFormBlock, blocks.length)}
                    <HotelsSearchInformationProvider />
                    {isDesktop ? (
                        <>
                            {this.renderBreadcrumbs()}
                            {this.renderDesktopContent()}
                            {this.renderFooterDisclaimer()}
                        </>
                    ) : (
                        <>
                            {this.renderMobileContent()}
                            {this.renderFooterDisclaimer()}
                            {this.renderBreadcrumbs()}
                        </>
                    )}
                </LayoutDefault>
            );
        }

        return null;
    }
}

export default GeoRegionPage;
