import {FC, useCallback, useEffect, useMemo, useState} from 'react';

import {
    IFullFilters,
    IChangeFilterGroupPayloadWithTarget,
    IChangePriceFilterPayloadWithTarget,
} from 'types/hotels/search/IFiltersInfo';
import {IWithClassName} from 'types/withClassName';
import {ISearchPageIdentifyParams} from 'types/hotels/geoRegion/IGeoRegionInfo';
import {EHotelsGoal} from 'utilities/metrika/types/goals/hotels';
import {IResetFilterAction} from 'server/api/HotelsSearchAPI/types/IResetFilterInfo';

import {
    getUpdatedFiltersByActiveAtoms,
    getUpdatedFiltersRangePrice,
} from 'reducers/hotels/searchPage/filters/utilities';

import {reachGoal} from 'utilities/metrika';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {hotelsURLs} from 'projects/hotels/utilities/urls';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {useExperiments} from 'utilities/hooks/useExperiments';
import {getFilterParams} from 'projects/hotels/utilities/getSearchPageFilterParams/getSearchPageFilterParams';
import {getSearchPageQueryByGeoRegionInfo} from 'projects/hotels/utilities/getSearchPageQueryByGeoRegionInfo/getSearchPageQueryByGeoRegionInfo';

import * as i18nBlock from 'i18n/hotels-GeoRegionPage';

import Flex from 'components/Flex/Flex';
import Text from 'components/Text/Text';
import FilterIcon from 'icons/16/Filter';
import Button from 'components/Button/Button';
import HotelsFilters from 'projects/hotels/components/HotelsFilters/HotelsFilters';

import cx from './Filters.scss';

interface IFiltersProps extends IWithClassName, IWithQaAttributes {
    filters: IFullFilters;
    searchParams: ISearchPageIdentifyParams;
    onResetFilter: (payload: IResetFilterAction) => void;
}

const Filters: FC<IFiltersProps> = props => {
    const deviceInfo = useDeviceType();
    const {
        className,
        filters: propsFilters,
        searchParams,
        onResetFilter,
    } = props;

    const experiments = useExperiments();
    const initialFilters = useMemo<IFullFilters>(
        () => ({
            ...propsFilters,
            foundHotelCount: 0,
            quickFilters: [],
            activeFilterAtoms: {},
        }),
        [propsFilters],
    );
    const [filters, setFilters] = useState<IFullFilters>(initialFilters);
    const [totalActiveFilters, setTotalActiveFilters] = useState(0);

    const renderFiltersBar = useCallback(
        ({onToggleViewFilters}: {onToggleViewFilters: () => void}) => (
            <Button
                className={cx('button')}
                size="xl"
                width="max"
                onClick={onToggleViewFilters}
            >
                <Flex justifyContent="space-between" alignItems="center">
                    <Text size="m">{i18nBlock.filtersButton()}</Text>
                    <FilterIcon />
                </Flex>
            </Button>
        ),
        [],
    );

    const handleRevertFilters = useCallback(() => {
        setFilters(initialFilters);
    }, [initialFilters]);

    const handleApplyFilters = useCallback(() => {
        reachGoal(EHotelsGoal.REGION_PAGE_FILTER_SUBMIT);
        window.open(
            hotelsURLs.getSearchUrlWithQuery(
                getSearchPageQueryByGeoRegionInfo({
                    ...searchParams,
                    filterParams: getFilterParams(filters),
                }),
            ),
        );
    }, [filters, searchParams]);

    const handleFilterGroupChange = useCallback(
        (payload: IChangeFilterGroupPayloadWithTarget) => {
            setFilters(getUpdatedFiltersByActiveAtoms(payload)(filters));
        },
        [filters],
    );

    const handlePriceFilterChange = useCallback(
        (payload: IChangePriceFilterPayloadWithTarget) => {
            setFilters(getUpdatedFiltersRangePrice(payload)(filters));
        },
        [filters],
    );

    const handleToggleViewFilters = useCallback(() => {
        reachGoal(EHotelsGoal.REGION_PAGE_FILTER_CLICK);
    }, []);

    useEffect(() => {
        // Настоящее значение кол-ва активных фильтров нам не важно
        // Важен факт их наличия (влияет на состояние кнопки сброса)
        setTotalActiveFilters(Object.keys(filters.activeFilterAtoms).length);
    }, [filters]);

    return (
        <HotelsFilters
            className={className}
            nights={1}
            filters={filters}
            deviceInfo={deviceInfo}
            canDisableFilters={false}
            experiments={experiments}
            totalActiveFilters={totalActiveFilters}
            renderFiltersBar={renderFiltersBar}
            onChangeFilterGroup={handleFilterGroupChange}
            onChangePriceFilter={handlePriceFilterChange}
            onResetFilters={handleRevertFilters}
            onRevertFilters={handleRevertFilters}
            onApplyFilters={handleApplyFilters}
            onToggleViewFilters={handleToggleViewFilters}
            onResetFilter={onResetFilter}
            {...prepareQaAttributes(props)}
        />
    );
};

export default Filters;
