import {Component, ReactNode} from 'react';

import {IWithDeviceType} from 'types/withDeviceType';
import {IHotelWithMinPrice} from 'types/hotels/hotel/IHotelWithMinPrice';
import {EHotelImageAltayStandardSize} from 'types/hotels/common/EHotelImageAltayStandardSize';
import {
    ESearchFormSize,
    ESearchFormTriggerViewType,
} from 'components/SearchForm/types';
import {EHotelsGoal} from 'utilities/metrika/types/goals/hotels';

import {parseImageUrlTemplate} from 'projects/hotels/utilities/prepareAndParseImages/prepareAndParseImages';
import {hotelsURLs} from 'projects/hotels/utilities/urls';
import {reachGoal} from 'utilities/metrika';
import {prepareQaAttributes} from 'utilities/qaAttributes/qaAttributes';

import * as i18nBlock from 'i18n/hotels-GeoRegionHotelSearchForm';

import BottomSheet from 'components/BottomSheet/BottomSheet';
import Modal, {EModalAnimationType} from 'components/Modal/Modal';
import TravelImage from 'components/TravelImage/TravelImage';
import Heading from 'components/Heading/Heading';
import Separator from 'components/Separator/Separator';
import TransportAccessibility from 'projects/hotels/components/TransportAccessibility/TransportAccessibility';
import HotelRatingAndCategory from 'projects/hotels/components/HotelRatingAndCategory/HotelRatingAndCategory';
import SearchForm from 'projects/hotels/components/SearchForm/SearchForm';

import cx from './GeoRegionHotelSearchForm.scss';

export interface IGeoRegionHotelSearchFormProps extends IWithDeviceType {
    canShowForm: boolean;
    activeHotel?: IHotelWithMinPrice;
    onHideForm: () => void;
}

/* Constants */
const DELAY_BEFORE_CHECK_CLOSE_BOTTOM_SHEET = 500;

const ROOT_QA = 'geoRegionHotelSearchFormModal';

class GeoRegionHotelSearchForm extends Component<IGeoRegionHotelSearchFormProps> {
    private _canCloseBottomSheet: boolean = true;

    /* Handlers */

    private handleSubmitForm = (): void => {
        reachGoal(EHotelsGoal.REGION_PAGE_SMALL_FORM_SUBMIT);

        this.hideForm();
    };

    private handleCloseModal = (): void => {
        this.hideForm();
    };

    private handleCloseBottomSheet = (): void => {
        if (this._canCloseBottomSheet) {
            this.hideForm();
        }
    };

    private handleShowFieldPopup = (): void => {
        this._canCloseBottomSheet = false;
    };

    private handleHideFieldPopup = (): void => {
        /* Prevent hide bottomSheet after close Modal */
        setTimeout(() => {
            this._canCloseBottomSheet = true;
        }, DELAY_BEFORE_CHECK_CLOSE_BOTTOM_SHEET);
    };

    /* Actions */

    private hideForm = (): void => {
        const {onHideForm} = this.props;

        onHideForm();
    };

    /* Render */

    private renderMobileSideSheet(): ReactNode {
        const {canShowForm} = this.props;

        return (
            <BottomSheet
                isOpened={canShowForm}
                className={cx('rootBottomSheet')}
                onClose={this.handleCloseBottomSheet}
                popHistoryOnUnmount={false}
            >
                {this.renderSearchForm()}
            </BottomSheet>
        );
    }

    private renderDesktopModal(): ReactNode {
        const {canShowForm} = this.props;

        return (
            <Modal
                isVisible={canShowForm}
                onClose={this.handleCloseModal}
                verticalAlign="top"
                animation={EModalAnimationType.NONE}
                containerClassName={cx('desktopModalContainer')}
            >
                <Modal.Content>{this.renderDesktopContent()}</Modal.Content>
            </Modal>
        );
    }

    private renderDesktopContent(): ReactNode {
        return (
            <div className={cx('desktopModal')}>
                {this.renderHotelCard()}
                <Separator className={cx('separator')} />
                <div className={cx('desktopModalTitle')}>
                    {i18nBlock.formTitle()}
                </div>
                {this.renderSearchForm()}
            </div>
        );
    }

    private renderHotelCard(): ReactNode {
        const {activeHotel} = this.props;

        if (!activeHotel) {
            return null;
        }

        const {hotel} = activeHotel;
        const {
            name,
            hotelSlug,
            nearestStations,
            images: [firstImage],
        } = hotel;
        const nearestStation = nearestStations?.[0];

        const hotelUrl = hotelsURLs.getHotelUrlByHotelSlug(hotelSlug);

        const imageUrl = firstImage
            ? parseImageUrlTemplate(
                  firstImage.urlTemplate,
                  EHotelImageAltayStandardSize.S,
                  firstImage.sizes,
              )
            : undefined;

        return (
            <div className={cx('hotelCard')}>
                <TravelImage src={imageUrl} className={cx('image')} />
                <div>
                    <Heading level={2}>{name}</Heading>
                    <HotelRatingAndCategory
                        hotelReviewsUrl={hotelUrl}
                        hotel={hotel}
                        size="m"
                    />
                    {nearestStation && (
                        <TransportAccessibility
                            station={nearestStation}
                            textWithIconSize="m"
                        />
                    )}
                </div>
            </div>
        );
    }

    private renderSearchForm = (): ReactNode => {
        const {
            activeHotel,
            deviceType: {isMobile, isDesktop},
        } = this.props;

        return (
            <SearchForm
                size={isMobile ? ESearchFormSize.XL : ESearchFormSize.S}
                triggerViewType={
                    isMobile
                        ? ESearchFormTriggerViewType.UNION
                        : ESearchFormTriggerViewType.TILE
                }
                hasDelayBeforeUpdateLocation
                hasDirectionSuggest={false}
                activeHotelSlug={activeHotel?.hotel?.hotelSlug}
                submitButtonText={i18nBlock.submitButtonNewTitle()}
                submitButtonTheme="primary"
                hideDateClearButton={isDesktop}
                canToggleDropdowns={isDesktop}
                onShowFieldPopup={this.handleShowFieldPopup}
                onHideFieldPopup={this.handleHideFieldPopup}
                onSubmit={this.handleSubmitForm}
                {...prepareQaAttributes({
                    parent: ROOT_QA,
                    current: 'searchForm',
                })}
            />
        );
    };

    render(): ReactNode {
        const {
            deviceType: {isMobile, isIe},
        } = this.props;

        /* Revert after fix Dropdown position */
        if (isIe) {
            return null;
        }

        if (isMobile) {
            return this.renderMobileSideSheet();
        }

        return this.renderDesktopModal();
    }
}

export default GeoRegionHotelSearchForm;
