import {PureComponent, ReactNode} from 'react';

import {IHotelsBlock} from 'types/hotels/geoRegion/IGeoRegionInfo';
import {IHotelWithMinPrice} from 'types/hotels/hotel/IHotelWithMinPrice';
import {IWithClassName} from 'types/withClassName';
import {IWithDeviceType} from 'types/withDeviceType';
import {EAdFoxBannerPosition, EAdFoxBannerType} from 'types/AdFox';

import {getSearchPageQueryByGeoRegionInfo} from 'projects/hotels/utilities/getSearchPageQueryByGeoRegionInfo/getSearchPageQueryByGeoRegionInfo';
import {hotelsURLs} from 'projects/hotels/utilities/urls';
import {deviceMods} from 'utilities/stylesUtils';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import Box from 'components/Box/Box';
import Heading from 'components/Heading/Heading';
import ButtonLink from 'components/ButtonLink/ButtonLink';
import AdFoxBanner from 'components/AdFoxBanner/AdFoxBanner';
import {ECardUsagePage} from 'projects/hotels/components/HotelInfoSearchCard/HotelInfoSearchCard';
import HotelWithMinPriceCard from 'projects/hotels/components/HotelCardWithMinPrice/HotelCardWithMinPrice';

import GeoRegionHotelSearchForm from '../GeoRegionHotelSearchForm/GeoRegionHotelSearchForm';

import cx from './HotelsBlock.scss';

interface IHotelsBlockProps
    extends IWithClassName,
        IWithDeviceType,
        IWithQaAttributes {
    block: IHotelsBlock;
    renderAdfox: boolean;
}

interface IHotelsBlockState {
    activeHotel?: IHotelWithMinPrice;
}

class HotelsBlock extends PureComponent<IHotelsBlockProps, IHotelsBlockState> {
    state = {
        activeHotel: undefined,
    };

    /* Handlers */
    private handleClickCardContent = (
        hotelWithMinPrice: IHotelWithMinPrice,
    ): void => {
        this.setState({activeHotel: hotelWithMinPrice});
    };

    private handleHideForm = (): void => {
        this.setState({activeHotel: undefined});
    };

    /* Helpers */
    private getSearchHotelsUrl = (): string => {
        const {
            block: {
                data: {searchParams},
            },
        } = this.props;
        const searchHotelsQueryParams =
            getSearchPageQueryByGeoRegionInfo(searchParams);

        return hotelsURLs.getSearchUrlWithQuery(searchHotelsQueryParams);
    };

    /* Render */
    private renderHotelWithMinPrice = (
        hotelWithMinPrice: IHotelWithMinPrice,
    ): ReactNode => {
        const {block} = this.props;
        const {
            hotel: {hotelSlug},
        } = hotelWithMinPrice;

        return (
            <HotelWithMinPriceCard
                key={hotelSlug}
                onClickHotelContent={this.handleClickCardContent}
                className={cx('hotelCard')}
                hotelWithMinPrice={hotelWithMinPrice}
                cardUsagePage={ECardUsagePage.REGION_PAGE}
                region={block.data.region}
                {...prepareQaAttributes({
                    parent: this.props,
                    current: 'hotelCard',
                    key: hotelSlug,
                })}
            />
        );
    };

    private redirectToSearchHotelsUrl = (): void => {
        const searchHotelsUrl = this.getSearchHotelsUrl();

        window.open(searchHotelsUrl, '_blank');
    };

    render(): ReactNode {
        const {
            className,
            deviceType,
            block: {
                data: {title, hotels, searchButtonText},
            },
            renderAdfox,
        } = this.props;

        const {activeHotel} = this.state;
        const {isDesktop} = deviceType;

        const ADFOX_PLACEMENT = 2;
        const leadingHotels = hotels.slice(0, ADFOX_PLACEMENT);
        const tailHotels = hotels.slice(ADFOX_PLACEMENT);

        return (
            <section className={cx('hotelsBlock', className)}>
                <Box below={deviceType.isDesktop ? 5 : 4}>
                    <Heading level={2}>{title}</Heading>
                </Box>

                <div className={cx(deviceMods('hotels', deviceType))}>
                    {leadingHotels.map(this.renderHotelWithMinPrice)}
                    {renderAdfox && (
                        <AdFoxBanner
                            className={cx('banner', {
                                banner_mobile: deviceType.isMobile,
                            })}
                            type={EAdFoxBannerType.Inline}
                            position={EAdFoxBannerPosition.Center}
                        />
                    )}
                    {tailHotels.map(this.renderHotelWithMinPrice)}
                </div>

                <ButtonLink
                    theme={isDesktop ? 'raised' : 'secondary'}
                    size="l"
                    target="_blank"
                    rel="noopener noreferrer"
                    width={isDesktop ? 'auto' : 'max'}
                    onClick={this.redirectToSearchHotelsUrl}
                >
                    {searchButtonText}
                </ButtonLink>

                <GeoRegionHotelSearchForm
                    deviceType={deviceType}
                    activeHotel={activeHotel}
                    canShowForm={Boolean(activeHotel)}
                    onHideForm={this.handleHideForm}
                />
            </section>
        );
    }
}

export default HotelsBlock;
