import React, {useCallback, useEffect, useMemo} from 'react';
import {useHistory, useLocation} from 'react-router-dom';
import {createPath} from 'history';
import {useDispatch} from 'react-redux';

import {EProjectName} from 'constants/common';
import {
    MC_HOTELS_VOUCHER_DOWNLOAD_PARAMS,
    MC_HOTELS_VOUCHER_PRINT_PARAMS,
} from 'constants/metrika/hotels';
import {PLUS_BONUS_POINTS_VESTING_PERIOD} from 'constants/plus';

import {EHotelsGoal} from 'utilities/metrika/types/goals/hotels';
import {EYandexPlusApplicationMode} from 'types/hotels/offer/IHotelOffer';
import {
    EDownloadActionTypes,
    ESearchByOrderId,
} from 'server/api/HotelsBookAPI/types/IDownloadVoucher';
import {IHappyPageHotelServiceResponse} from 'server/services/OrdersService/types/TGetOrderHappyPageServiceResponse';
import {EHotelOrderPaymentType} from 'server/api/HotelsBookAPI/types/IOrder';

import {setIsNewTrip} from 'reducers/trips/tripPage/actions';

import {getOrderUrl} from 'projects/account/utilities/urls/getOrderUrl';
import {getHotelsVoucherUrl} from 'projects/hotels/utilities/getHotelsVoucherUrl/getHotelsVoucherUrl';
import {hotelsURLs} from 'projects/hotels/utilities/urls';
import {reachGoal} from 'utilities/metrika';
import {useReachGoal} from 'utilities/metrika/useReachGoal';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {deviceMods} from 'utilities/stylesUtils';
import {prepareQaAttributes} from 'utilities/qaAttributes/qaAttributes';
import {prepareDeferredPaymentDateTime} from 'projects/hotels/utilities/prepareDeferredPaymentDate/prepareDeferredPaymentDate';
import {HUMAN_DATETIME_WITHOUT_YEAR} from 'utilities/dateUtils/formats';
import {useHotelVoucherStatus} from 'utilities/hooks/useHotelVoucherStatus';

import * as i18nBlock from 'i18n/happyPage';
import * as i18nHotelsPlusBlock from 'i18n/hotels-plusPromo2021';

import HappyPageContentBase from 'src/projects/happyPage/components/HappyPageContent/HappyPageContent';
import OrderInfo from 'projects/hotels/pages/HappyPage/components/OrderInfo/OrderInfo';
import OrderActions, {
    EOrderActionsSource,
} from 'components/OrderActions/OrderActions';
import Separator from 'components/Separator/Separator';
import PlusBanner from 'projects/happyPage/components/PlusBanner/PlusBanner';
import NextPaymentInfo from 'projects/hotels/pages/HappyPage/components/NextPaymentInfo/NextPaymentInfo';
import Spinner from 'components/Spinner/Spinner';
import CardWithDeviceLayout from 'components/CardWithDeviceLayout/CardWithDeviceLayout';
import WhiteLabelPartnerBanner from 'projects/hotels/pages/HappyPage/components/WhiteLabelPartnerBanner/WhiteLabelPartnerBanner';

import cx from './HappyPageContent.scss';

const HAPPY_PAGE_QA = 'happyPage';

interface IHappyPageContentProps {
    orderId: string;
    orderAndCrossSalesInfo: IHappyPageHotelServiceResponse;
}

const HappyPageContent: React.FC<IHappyPageContentProps> = props => {
    const {orderId, orderAndCrossSalesInfo} = props;

    const location = useLocation();
    const history = useHistory();
    const deviceType = useDeviceType();
    const dispatch = useDispatch();

    const {order} = orderAndCrossSalesInfo;
    const {
        confirmationInfo: {yandexOrderId},
        payment: {
            current: currentPayment,
            next: nextPayment,
            usesDeferredPayments,
        },
        orderPriceInfo,
        status,
        orderCancellationDetails,
        guestsInfo: {customerEmail},
        appliedPromoCampaigns,
    } = order;

    const plusInfo = appliedPromoCampaigns?.yandexPlus;

    const title =
        usesDeferredPayments && !nextPayment
            ? i18nBlock.successTextTitleHotelFullyPaid()
            : i18nBlock.successTextTitleHotel();

    const {
        isReady: isVoucherReady,
        isLoading: isVoucherStatusLoading,
        isError: isVoucherStatusError,
    } = useHotelVoucherStatus(orderId);

    const reachOrderShownGoal = useReachGoal(
        EHotelsGoal.HOTEL_HAPPY_PAGE_ORDER_SHOWN,
    );
    const reachDetailsClick = useReachGoal(
        EHotelsGoal.HOTEL_HAPPY_PAGE_ORDER_DETAILS_CLICK,
    );
    const reachPrintClick = useReachGoal(
        EHotelsGoal.HAPPY_PAGE_VOUCHER_BUTTON,
        [MC_HOTELS_VOUCHER_PRINT_PARAMS],
    );
    const reachDownloadClick = useReachGoal(
        EHotelsGoal.HAPPY_PAGE_VOUCHER_BUTTON,
        [MC_HOTELS_VOUCHER_DOWNLOAD_PARAMS],
    );

    const banner = useMemo(() => {
        const isPlusTopup = Boolean(
            plusInfo?.mode === EYandexPlusApplicationMode.TOPUP &&
                plusInfo?.points,
        );
        const partnerInfo = orderPriceInfo?.promoCampaigns.whiteLabel;

        return (
            <>
                {isPlusTopup && (
                    <PlusBanner
                        className={cx(
                            'plusBanner',
                            deviceMods('plusBanner', deviceType),
                        )}
                        deviceType={deviceType}
                        text={i18nHotelsPlusBlock.happyPageBannerText({
                            days: PLUS_BONUS_POINTS_VESTING_PERIOD,
                            plusPoints: plusInfo?.points || 0,
                        })}
                    />
                )}
                <WhiteLabelPartnerBanner
                    points={partnerInfo?.points}
                    deviceType={deviceType}
                />
            </>
        );
    }, [deviceType, orderPriceInfo?.promoCampaigns.whiteLabel, plusInfo]);

    const nextPaymentInfo = useMemo(() => {
        if (!nextPayment) {
            return null;
        }

        return (
            <>
                <NextPaymentInfo
                    url={getOrderUrl(orderId)}
                    className={cx(
                        'nextPaymentInfo',
                        deviceMods('nextPaymentInfo', deviceType),
                    )}
                    deviceType={deviceType}
                    price={nextPayment.amount}
                    dateText={prepareDeferredPaymentDateTime(
                        nextPayment.paymentEndsAt,
                        HUMAN_DATETIME_WITHOUT_YEAR,
                    )}
                    {...prepareQaAttributes(HAPPY_PAGE_QA)}
                />
                {deviceType.isMobile && <Separator margin={4} />}
            </>
        );
    }, [deviceType, nextPayment, orderId]);

    const orderInfo = useMemo(() => {
        return (
            <CardWithDeviceLayout radius={deviceType.isMobile ? 'm' : 'l'}>
                <OrderInfo
                    order={order}
                    onMount={reachOrderShownGoal}
                    {...prepareQaAttributes(HAPPY_PAGE_QA)}
                />
            </CardWithDeviceLayout>
        );
    }, [deviceType.isMobile, order, reachOrderShownGoal]);

    const renderSpinnerIcon = useCallback(
        (className?: string): JSX.Element => (
            <Spinner className={className} size="xxs" />
        ),
        [],
    );

    const orderActions = useMemo(() => {
        return (
            <OrderActions
                source={EOrderActionsSource.HAPPY_PAGE}
                orderDetails={{
                    url: getOrderUrl(orderId),
                    onClick: reachDetailsClick,
                }}
                print={{
                    title: i18nBlock.print(),
                    url: getHotelsVoucherUrl(yandexOrderId, {
                        actionType: EDownloadActionTypes.PRINT,
                        searchBy: ESearchByOrderId.PRETTY_ORDER_ID,
                    }),
                    renderIcon: isVoucherStatusLoading
                        ? renderSpinnerIcon
                        : undefined,
                    isDisabled: !isVoucherReady,
                    onClick: reachPrintClick,
                }}
                download={{
                    title: i18nBlock.download(),
                    url: getHotelsVoucherUrl(yandexOrderId, {
                        actionType: EDownloadActionTypes.DOWNLOAD,
                        searchBy: ESearchByOrderId.PRETTY_ORDER_ID,
                    }),
                    renderIcon: isVoucherStatusLoading
                        ? renderSpinnerIcon
                        : undefined,
                    isDisabled: !isVoucherReady,
                    onClick: reachDownloadClick,
                }}
                {...prepareQaAttributes({
                    parent: HAPPY_PAGE_QA,
                    current: 'orderActions',
                })}
            />
        );
    }, [
        orderId,
        reachDetailsClick,
        isVoucherStatusLoading,
        renderSpinnerIcon,
        isVoucherReady,
        reachPrintClick,
        reachDownloadClick,
        yandexOrderId,
    ]);

    useEffect(() => {
        const currentPath = createPath(location);
        const path = hotelsURLs.getPathByOrderStatus({
            orderId,
            orderStatus: status,
            orderCancellationDetails,
            location,
        });

        if (path && currentPath !== path) {
            history.replace(path);
        }
    }, [history, location, orderCancellationDetails, orderId, status]);

    useEffect(() => {
        if (isVoucherReady) {
            reachGoal(EHotelsGoal.HAPPY_PAGE_VOUCHER_SUCCESS);
        }
    }, [isVoucherReady]);

    useEffect(() => {
        if (isVoucherStatusError) {
            reachGoal(EHotelsGoal.HAPPY_PAGE_VOUCHER_ERROR);
        }
    }, [isVoucherStatusError]);

    useEffect(() => {
        // FIXME: Временное решение пока нет нотификаций: Показываем уведомлление о новой поездке только сразу после создания заказа, а не при доплатах
        if (
            !currentPayment ||
            currentPayment.type === EHotelOrderPaymentType.FULL ||
            currentPayment.type === EHotelOrderPaymentType.INITIAL
        ) {
            dispatch(setIsNewTrip(true));
        }
    }, [currentPayment, dispatch]);

    return (
        <HappyPageContentBase
            orderType={EProjectName.HOTELS}
            banner={banner}
            orderHeaderInfo={nextPaymentInfo}
            orderInfo={orderInfo}
            orderActions={orderActions}
            crossSale={orderAndCrossSalesInfo.crossSale}
            prettyOrderId={yandexOrderId}
            email={customerEmail}
            title={title}
            {...prepareQaAttributes(HAPPY_PAGE_QA)}
        />
    );
};

export default React.memo(HappyPageContent);
