import React, {useEffect} from 'react';

import {IOrder} from 'server/api/HotelsBookAPI/types/IOrder';

import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {deviceMods} from 'utilities/stylesUtils';
import {
    IWithDataQaAttribute,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {CHAR_NBSP} from 'utilities/strings/charCodes';

import * as i18nBlockAccount from 'i18n/account-OrderHotels';

import Heading from 'components/Heading/Heading';
import HotelAddress from 'projects/hotels/components/HotelAddress/HotelAddress';
import BookSearchParams from 'projects/hotels/components/BookSearchParams/BookSearchParams';
import Box from 'components/Box/Box';
import Flex from 'components/Flex/Flex';
import Image from 'components/TravelImage/TravelImage';
import Text from 'components/Text/Text';
import Separator from 'components/Separator/Separator';
import HotelStars from 'components/HotelStars/HotelStars';

import cx from './OrderInfo.scss';

interface IOrderInfoProps extends IWithDataQaAttribute {
    order: IOrder;
    onMount?(): void;
}

const ORDER_INFO_QA = 'orderInfo';

const OrderInfo: React.FC<IOrderInfoProps> = props => {
    const {
        order: {
            offerInfo: {
                hotelInfo,
                searchParams,
                hotelInfo: {name, stars, address, partnerHotelName},
                roomInfo: {
                    images: [mainImage],
                },
            },
        },
        onMount,
    } = props;

    const deviceType = useDeviceType();

    useEffect(() => {
        onMount?.();
    }, [onMount]);

    const roomImageUrl = mainImage?.links?.['350px']?.href;

    const hotelImage = roomImageUrl ? (
        <Image
            className={cx('imageWrapper')}
            src={roomImageUrl}
            isWide
            isCrop
        />
    ) : null;

    const partnerName = (
        <Text
            className={cx('partnerName')}
            size="xs"
            color="secondary"
            tag="div"
        >
            {i18nBlockAccount.partnerName({
                hotelName: partnerHotelName,
            })}
        </Text>
    );

    return (
        <Box
            className={cx('root', deviceMods('root', deviceType))}
            {...prepareQaAttributes({
                parent: props,
                current: ORDER_INFO_QA,
            })}
        >
            <Flex justifyContent="space-between">
                <Box className={cx('mainHotelInfo')}>
                    <Flex justifyContent="space-between">
                        <Heading
                            level={deviceType.isMobile ? 3 : 1}
                            {...prepareQaAttributes({
                                parent: props,
                                current: `${ORDER_INFO_QA}-hotelName`,
                            })}
                        >
                            {name}
                            {CHAR_NBSP}
                            <HotelStars
                                stars={stars}
                                size={deviceType.isMobile ? '12' : '16'}
                            />
                        </Heading>

                        {deviceType.isMobile && hotelImage}
                    </Flex>

                    <HotelAddress
                        className={cx('address')}
                        type="text"
                        text={address}
                        multiLine
                    />

                    <BookSearchParams
                        className={cx('dates')}
                        hotelInfo={hotelInfo}
                        searchParams={searchParams}
                        showChangeParamsLink={false}
                        showGuestsParams={false}
                        {...prepareQaAttributes({
                            parent: props,
                            current: ORDER_INFO_QA,
                        })}
                    />
                </Box>

                {deviceType.isDesktop && (
                    <Flex
                        className={cx('photoWithPartnerName')}
                        flexDirection="column"
                        between={2}
                    >
                        {hotelImage}

                        {deviceType.isDesktop && partnerName}
                    </Flex>
                )}
            </Flex>

            {deviceType.isMobile && (
                <>
                    <Separator className={cx('separator')} />

                    {partnerName}
                </>
            )}
        </Box>
    );
};

export default React.memo(OrderInfo);
