import React, {PureComponent, ReactNode} from 'react';
import _omit from 'lodash/omit';

import {ERedirectStatusCodes} from 'constants/redirectStatusCodes';
import {URLs} from 'constants/urls';
import {DEFAULT_ADULTS_COUNT} from 'constants/hotels';

import {IHotelPageQueryParams} from 'types/hotels/common/IQueryParams';
import {EHotelsGoal} from 'utilities/metrika/types/goals/hotels';

import {IDevice} from 'reducers/common/commonReducerTypes';

import {reachGoal} from 'utilities/metrika';
import {internalUrl} from 'utilities/url';
import {deviceMods} from 'utilities/stylesUtils';
import getQueryByLocation from 'utilities/getQueryByLocation/getQueryByLocation';
import {getOfferRequiredSearchParamsByLocation} from 'projects/hotels/utilities/getHotelPageParams/getOfferRequiredSearchParamsByLocation';
import {getHotelInfoSearchParamsByLocation} from 'projects/hotels/utilities/getHotelPageParams/getHotelInfoSearchParamsByLocation';
import {getViewModePageParamsByQuery} from 'projects/hotels/utilities/getHotelPageParams/getHotelPageParams';
import {getHotelSlugByRouteMatch} from 'projects/hotels/utilities/getHotelIdentifier/getHotelIdentifier';
import {prepareQaAttributes} from 'utilities/qaAttributes/qaAttributes';
import {hotelsURLs} from 'projects/hotels/utilities/urls';
import {getHotelTaxiPromocodeBadge} from 'projects/hotels/utilities/getHotelBadges/getHotelTaxiPromocodeBadge';
import {loadable} from 'utilities/componentLoadable';
import sendMetrikaExtraVisitAndUserParams from 'projects/hotels/utilities/metrika/sendMetrikaExtraVisitAndUserParams';

import * as i18nBlock from 'i18n/page-error-boundary';

import SchemaMarkup, {
    getHotelSchemaMarkup,
} from 'components/SchemaMarkup/SchemaMarkup';
import PageLoader from 'components/BookLoader/BookLoader';
import TaxiDisclaimer from 'components/TaxiDisclaimer/TaxiDisclaimer';
import ChunksPreloader from 'components/ChunksPreloader/ChunksPreloader';
import NotFoundPage from 'components/NotFound/NotFoundPage/NotFoundPage';
import LayoutDefault from 'components/Layouts/LayoutDefault/LayoutDefault';
import LayoutError500 from 'components/Layouts/LayoutError500/LayoutError500';
import RedirectWithStatus from 'components/RedirectWithStatus/RedirectWithStatus';
import SpecifyDatesWarning from './components/SpecifyDatesWarning/SpecifyDatesWarning';
import HotelPageSeoBreadCrumbs from './components/HotelPageSeoBreadCrumbs/HotelPageSeoBreadCrumbs';
import HotelPageLayout from 'projects/hotels/pages/HotelPage/components/HotelPageLayout/HotelPageLayout';
import HotelPageSearchController from './components/HotelPageSearchController/HotelPageSearchController';
import HotelPageCard from 'projects/hotels/pages/HotelPage/components/HotelPageCard/HotelPageCardContainer';
import HangingYandexMessenger from 'components/YandexMessenger/components/HangingYandexMessenger/HangingYandexMessenger';

import HotelsSearchInformationProvider from 'projects/hotels/containers/HotelsSearchInformationProvider/HotelsSearchInformationProvider';

import DeviceTypeContext from 'contexts/DeviceTypeContext';

import {IHotelPageContainer} from './HotelPageContainer';

import cx from './HotelPage.scss';

const HOTEL_PAGE_QA = 'hotels-hotelPage';
const PRELOAD_CHUNKS = [
    loadable(() => import('projects/hotels/pages/BookPage/BookPageContainer')),
];

interface IHotelPageProps extends IHotelPageContainer {}

class HotelPage extends PureComponent<IHotelPageProps> {
    componentDidMount(): void {
        const {hotelInfo} = this.props;

        if (hotelInfo.data?.extraVisitAndUserParams)
            sendMetrikaExtraVisitAndUserParams(
                hotelInfo.data?.extraVisitAndUserParams,
            );

        reachGoal(EHotelsGoal.HOTEL_PAGE_BOY_MAIN_TAB);
        reachGoal(EHotelsGoal.HOTEL_PAGE_VISITED);
    }

    /* Helpers */

    private isHeadless(): boolean | undefined {
        const {location} = this.props;
        const queryByLocation = getQueryByLocation(location);
        const {isHeadless} = getViewModePageParamsByQuery(queryByLocation);

        return isHeadless;
    }

    private checkRedirectToHotelPageWithActualSlug(): boolean {
        if (__CLIENT__) {
            return false;
        }

        const {match, hotelInfo} = this.props;
        const hotelSlugByResponse = hotelInfo?.data?.hotel?.hotelSlug;
        const hotelSlugByRouteMatch = getHotelSlugByRouteMatch(match);

        return (
            hotelInfo.isSuccess &&
            Boolean(hotelSlugByResponse) &&
            hotelSlugByResponse !== hotelSlugByRouteMatch
        );
    }

    private getHotelPageRoutePathWithSlug(): string {
        const {hotelInfo, location} = this.props;
        const hotelSlugByResponse = hotelInfo?.data?.hotel?.hotelSlug;
        const queryByLocation: IHotelPageQueryParams =
            getQueryByLocation(location);
        const queryByLocationWithoutPermalink = _omit<
            IHotelPageQueryParams,
            'hotelPermalink'
        >(queryByLocation, 'hotelPermalink');

        if (hotelSlugByResponse) {
            return hotelsURLs.getHotelPageWithSlug(
                hotelSlugByResponse,
                queryByLocationWithoutPermalink,
            );
        }

        return internalUrl(URLs.notFound);
    }

    private goToNotFound(): void {
        const {history} = this.props;

        history.push(internalUrl(URLs.notFound));
    }

    /* Fetch data */

    private fetchHotelInfo(): void {
        const {
            location,
            match: {
                params: {regionSlug, hotelSlug},
            },
            getHotelInfo,
        } = this.props;

        const hotelInfoSearchParams = getHotelInfoSearchParamsByLocation({
            queryParams: getQueryByLocation(location),
            regionSlug,
            hotelSlug,
        });

        if (
            hotelInfoSearchParams.permalink ||
            hotelInfoSearchParams.hotelSlug
        ) {
            getHotelInfo(hotelInfoSearchParams);
        } else {
            this.goToNotFound();
        }
    }

    private fetchSimilarHotels(): void {
        const {getSimilarHotels} = this.props;

        getSimilarHotels();
    }

    private fetchOffers(): void {
        const {getHotelInfoOffers} = this.props;

        getHotelInfoOffers();
    }

    private fetchAdditionalHotelInfo(): void {
        const {getAdditionalHotelInfo} = this.props;

        getAdditionalHotelInfo();
    }

    private stopCurrentHotelInfoActions(): void {
        const {stopHotelInfoActions} = this.props;

        stopHotelInfoActions();
    }

    private updateHotelInfoSearchParams(): void {
        const {setHotelInfoSearchParams} = this.props;
        const offerRequiredSearchParams =
            getOfferRequiredSearchParamsByLocation();

        if (offerRequiredSearchParams) {
            setHotelInfoSearchParams(offerRequiredSearchParams);
        }
    }

    /* Handlers */

    private handleChangeHotelIdentifier = (): void => {
        this.fetchHotelInfo();
    };

    private handleChangeLocationSearchParams = (): void => {
        this.updateHotelInfoSearchParams();
    };

    private handleChangeHotelInfoSearchParams = (): void => {
        this.fetchOffers();
        this.fetchSimilarHotels();
    };

    private handleStartFetchAdditionalInfo = (): void => {
        this.fetchAdditionalHotelInfo();
    };

    private handleStopHotelInfoActions = (): void => {
        this.stopCurrentHotelInfoActions();
    };

    private handleSearchParamsRemoved = (): void => {
        const {
            resetHotelInfoToInitial,
            resetHotelsSearchForm,
            updateSearchInformation,
        } = this.props;

        resetHotelInfoToInitial();
        resetHotelsSearchForm();
        updateSearchInformation({
            childrenAges: [],
            adults: DEFAULT_ADULTS_COUNT,
            checkinDate: '',
            checkoutDate: '',
        });
        this.fetchHotelInfo();
    };

    /* Render */

    private renderPageLoader(): ReactNode {
        const {
            hotelInfo: {isLoading},
        } = this.props;

        return (
            <PageLoader
                isLoading={isLoading}
                isModalView={true}
                title="Загрузка отеля"
                description="Загружаем информацию об отеле и получаем самые выгодные предложения"
            />
        );
    }

    private renderPreloadChunksControl(): ReactNode {
        return (
            <ChunksPreloader
                preloadChunks={PRELOAD_CHUNKS}
                maxProportionTimeout={20000}
            />
        );
    }

    private renderHotelInfo(deviceType: IDevice): ReactNode {
        const {
            location,
            hotelInfo: {isSuccess, data, isLoadingOffers},
        } = this.props;
        const isHeadless = this.isHeadless();

        if (!isSuccess || !data) {
            return null;
        }

        return (
            <HotelPageCard
                className={cx('hotelCard')}
                location={location}
                hotelInfo={data}
                isLoadingOffers={isLoadingOffers}
                deviceType={deviceType}
                isHeadless={Boolean(isHeadless)}
            />
        );
    }

    private renderSearchInformationProvider(): ReactNode {
        return <HotelsSearchInformationProvider />;
    }

    private renderHotelSearchController(): ReactNode {
        const {
            match,
            location,
            hotelInfo,
            resetHotelInfoToInitial,
            resetHotelsSearchForm,
        } = this.props;
        const hotelSlug = hotelInfo?.data?.hotel?.hotelSlug;
        const permalink = hotelInfo?.data?.hotel?.permalink;
        const searchParams = hotelInfo?.data?.searchParams;

        return (
            <HotelPageSearchController
                location={location}
                match={match}
                hotelSlug={hotelSlug}
                permalink={permalink}
                searchParams={searchParams}
                onStartFetchAdditionalInfo={this.handleStartFetchAdditionalInfo}
                onStopHotelInfoActions={this.handleStopHotelInfoActions}
                onChangeLocationSearchParams={
                    this.handleChangeLocationSearchParams
                }
                onChangeHotelInfoSearchParams={
                    this.handleChangeHotelInfoSearchParams
                }
                onChangeHotelIdentifier={this.handleChangeHotelIdentifier}
                resetHotelInfoToInitial={resetHotelInfoToInitial}
                resetHotelSearchForm={resetHotelsSearchForm}
                onSearchParamsRemoved={this.handleSearchParamsRemoved}
            />
        );
    }

    private renderSeoBreadcrumbs(): ReactNode {
        const {
            hotelInfo: {data},
            schemaNonce,
        } = this.props;

        if (!data) {
            return null;
        }

        const {seoBreadcrumbs} = data;

        return (
            <HotelPageSeoBreadCrumbs
                breadcrumbs={seoBreadcrumbs}
                nonce={schemaNonce}
            />
        );
    }

    private renderDisclaimer(): ReactNode {
        const {
            hotelInfo: {data},
        } = this.props;
        const hasTaxiBadge = data?.offersInfo?.mainOffers?.some(offer =>
            getHotelTaxiPromocodeBadge(offer.badges),
        );

        return hasTaxiBadge ? (
            <div className={cx('disclaimer')}>
                <TaxiDisclaimer />
            </div>
        ) : null;
    }

    private renderWarning(deviceType: IDevice): ReactNode {
        const {
            hotelInfo: {data},
        } = this.props;

        if (deviceType.isDesktop || data?.searchParams) {
            return null;
        }

        return <SpecifyDatesWarning className={cx('specifyDatesWarning')} />;
    }

    private renderContent = (deviceType: IDevice): ReactNode => {
        const isHeadless = this.isHeadless();

        return (
            <div
                className={cx(
                    {content_isHeadless: isHeadless},
                    deviceMods('content', deviceType),
                )}
                {...prepareQaAttributes(HOTEL_PAGE_QA)}
            >
                {isHeadless ? (
                    <>
                        {this.renderPageLoader()}
                        {this.renderPreloadChunksControl()}
                        {this.renderSearchInformationProvider()}
                        {this.renderHotelSearchController()}
                        {this.renderHotelInfo(deviceType)}
                    </>
                ) : (
                    <>
                        {this.renderWarning(deviceType)}
                        {this.renderPageLoader()}
                        {this.renderPreloadChunksControl()}
                        {this.renderSearchInformationProvider()}
                        {this.renderHotelSearchController()}
                        {this.renderHotelInfo(deviceType)}
                        {this.renderDisclaimer()}
                        {this.renderSeoBreadcrumbs()}
                        {this.renderMessenger()}
                    </>
                )}
            </div>
        );
    };

    private renderMessenger(): ReactNode {
        return (
            <HangingYandexMessenger
                entrypoint="hotelPage"
                metrikaGoal={EHotelsGoal.HOTELS_CHAT_CLICK}
            />
        );
    }

    private renderSchemaMarkup(): ReactNode {
        const {hotelInfo, schemaNonce} = this.props;

        if (!hotelInfo.data) {
            return null;
        }

        const data = getHotelSchemaMarkup(hotelInfo.data);

        return <SchemaMarkup data={data} nonce={schemaNonce} />;
    }

    render(): ReactNode {
        const isHeadless = this.isHeadless();
        const canRedirectToHotelPageWithActualSlug =
            this.checkRedirectToHotelPageWithActualSlug();
        const {
            hotelInfo: {data, isError, status},
        } = this.props;

        if (isError) {
            switch (status) {
                case 404:
                    return <NotFoundPage />;
                default:
                    return (
                        <LayoutDefault>
                            <LayoutError500
                                title={i18nBlock.title()}
                                subtitle={i18nBlock.subtitle()}
                                action={{
                                    type: 'button',
                                    title: i18nBlock.actionDashTitle(),
                                    handler: (): void =>
                                        window.document.location.reload(),
                                }}
                            />
                        </LayoutDefault>
                    );
            }
        }

        if (canRedirectToHotelPageWithActualSlug) {
            const hotelPagePath = this.getHotelPageRoutePathWithSlug();

            return (
                <RedirectWithStatus
                    to={hotelPagePath}
                    statusCode={ERedirectStatusCodes.PERMANENTLY}
                />
            );
        }

        return (
            <DeviceTypeContext.Consumer>
                {(deviceType): React.ReactNode => (
                    <HotelPageLayout
                        isHeaderShown={!isHeadless}
                        isFooterShown={!isHeadless}
                        searchFormCollapsedClassName={cx({
                            searchTriggerWithoutDates:
                                deviceType.isMobile && !data?.searchParams,
                        })}
                    >
                        {this.renderSchemaMarkup()}
                        {this.renderContent(deviceType)}
                    </HotelPageLayout>
                )}
            </DeviceTypeContext.Consumer>
        );
    }
}

export default HotelPage;
