import React, {FC, useMemo} from 'react';
import {RouteComponentProps, withRouter} from 'react-router-dom';

import {EAppActions} from 'constants/platforms/TPlatforms';

import {IGeoRegion} from 'types/hotels/hotel/IGeoRegion';
import {IWithClassName} from 'types/withClassName';

import {deviceMods} from 'utilities/stylesUtils';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import useQuery from 'utilities/hooks/useQuery';

import * as commonKeyset from 'i18n/common';
import * as i18nBlockPage from 'i18n/hotels-HotelPage';

import Link from 'components/Link/Link';
import ArrowLeftIcon from 'icons/16/ArrowLeft';

import {useCoordinator, usePlatform} from 'contexts/PlatformContext';

import cx from './BackButton.scss';

export interface IBackButtonProps
    extends RouteComponentProps,
        IWithQaAttributes,
        IWithClassName {
    backGeoRegionLink?: string;
    backGeoRegion?: IGeoRegion;
}

const BackButton: FC<IBackButtonProps> = props => {
    const {history, backGeoRegion, backGeoRegionLink, className} = props;
    const {isWeb} = usePlatform();
    const deviceType = useDeviceType();
    const coordinator = useCoordinator();
    const {fromPage} = useQuery();

    const buttonLinkProps = useMemo(() => {
        if (!isWeb) {
            const onClick: React.MouseEventHandler = e => {
                const processed = coordinator.doAction(
                    fromPage === 'favorites'
                        ? EAppActions.NAVIGATE_TO_FAVORITES
                        : EAppActions.NAVIGATE_TO_SEARCH_HOTELS_PAGE,
                );

                if (processed) {
                    e.preventDefault();

                    return;
                }
            };

            return {
                onClick,
            };
        }

        if (history.location.state && history.location.state.goBack) {
            return {
                to: undefined,
                onClick: history.goBack,
            };
        }

        return {
            to: backGeoRegionLink,
        };
    }, [
        isWeb,
        history.location.state,
        history.goBack,
        backGeoRegionLink,
        coordinator,
        fromPage,
    ]);

    const geoRegionGenitiveCase = backGeoRegion?.linguistics?.genitiveCase;

    if (!backGeoRegionLink || !geoRegionGenitiveCase) {
        return null;
    }

    const backButtonText =
        !isWeb && fromPage === 'favorites'
            ? commonKeyset.back()
            : `${i18nBlockPage.geoRegionPrefix()} ${geoRegionGenitiveCase}`;

    return (
        <Link
            className={cx(
                'backButton',
                deviceMods('backButton', deviceType),
                {
                    backButton_web: isWeb,
                },
                className,
            )}
            theme="black"
            {...buttonLinkProps}
            {...prepareQaAttributes({
                current: 'backButton',
                parent: props,
            })}
        >
            <ArrowLeftIcon className={cx('backButtonArrow')} />
            {backButtonText}
        </Link>
    );
};

export default withRouter(BackButton);
