import React, {useMemo} from 'react';

import {
    IAggregatedOfferInfo,
    IRequiredOfferParams,
} from 'types/hotels/offer/IHotelOffer';
import {IHotelOffersInfo} from 'server/api/HotelSearchAPI/types/IOffersInfo';

import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {getTotalNights} from 'projects/hotels/utilities/calculateTotalNights/calculateTotalNights';
import {CHAR_NBSP} from 'utilities/strings/charCodes';
import checkOffersDiscount from '../../utilities/checkOffersDiscount';

import * as i18nBlock from 'i18n/hotels-HotelPage';

import Price from 'components/Price/Price';
import Button from 'components/Button/Button';
import OfferLabels from 'projects/hotels/pages/HotelPage/components/OffersInfo/components/OfferLabels/OfferLabels';
import SellIcon from 'icons/24/Sell';

import cx from './CommonOffersInfo.scss';

interface ICommonOffersInfo extends IWithQaAttributes {
    aggregatedOfferInfo: IAggregatedOfferInfo;
    offersInfo: IHotelOffersInfo;
    onAllOffersClick: () => void;
    searchParams: IRequiredOfferParams;
}

const CommonOffersInfo: React.FC<ICommonOffersInfo> = props => {
    const {
        offersInfo: {mainOffers, partnerOffers},
        onAllOffersClick,
        searchParams,
        aggregatedOfferInfo: {
            minPrice,
            maxPrice,
            pansionAggregate,
            cancellationInfoAggregate,
        },
    } = props;

    const hasDiscount = useMemo(
        () => checkOffersDiscount(mainOffers, partnerOffers),
        [mainOffers, partnerOffers],
    );

    const nightsCount = getTotalNights(
        searchParams.checkinDate,
        searchParams.checkoutDate,
    );

    return (
        <div className={cx('root')}>
            <div
                className={cx('price')}
                {...prepareQaAttributes({
                    parent: props,
                    current: 'price',
                })}
            >
                {minPrice.value === maxPrice.value &&
                minPrice.currency === maxPrice.currency ? (
                    <Price
                        value={minPrice.value}
                        currency={minPrice.currency}
                    />
                ) : (
                    <>
                        <Price
                            value={minPrice.value}
                            currency={minPrice.currency}
                            isFrom={true}
                        />{' '}
                        <Price
                            value={maxPrice.value}
                            currency={maxPrice.currency}
                            isUpTo={true}
                        />
                        {hasDiscount && (
                            <>
                                {CHAR_NBSP}
                                <SellIcon className={cx('priceDiscountIcon')} />
                            </>
                        )}
                    </>
                )}
            </div>
            <div
                className={cx('nights')}
                {...prepareQaAttributes({parent: props, current: 'nights'})}
            >
                {i18nBlock.nightsCount({nightsCount})}
            </div>
            <OfferLabels
                className={cx('labels')}
                mealType={pansionAggregate}
                cancellationType={cancellationInfoAggregate}
                {...prepareQaAttributes({
                    parent: props,
                    current: 'offerLabels',
                })}
            />
            <Button
                className={cx('offerBookLink')}
                onClick={onAllOffersClick}
                size="l"
                theme="primary"
                {...prepareQaAttributes({
                    parent: props,
                    current: 'allOffersButton',
                })}
            >
                {mainOffers && mainOffers.length > 0
                    ? i18nBlock.boyViewAllOffersAction()
                    : i18nBlock.metaViewAllOffersAction()}
            </Button>
        </div>
    );
};

export default CommonOffersInfo;
