import React, {useCallback} from 'react';

import {IWithClassName} from 'types/withClassName';

import defaultRenderTotalCount from './utilities/defaultRenderTotalCount';

import Box from 'components/Box/Box';
import Image from 'components/TravelImage/TravelImage';

import cx from './GalleryRow.scss';

interface IGalleryRowProps extends IWithClassName {
    imageWrapperClassName?: string;
    imageSources: string[];
    isLastClickable?: boolean;
    totalImageCount?: number;
    imagesAlt?: string;
    renderTotalCount?(totalCount?: number): React.ReactNode;
    onImageClick(imageIndex: number): void;
    onMoreImagesClick?(imageIndex: number): void;
}

const GalleryRow: React.FC<IGalleryRowProps> = props => {
    const {
        imageSources,
        totalImageCount,
        onMoreImagesClick,
        isLastClickable,
        className,
        imageWrapperClassName,
        onImageClick,
        imagesAlt,
        renderTotalCount = defaultRenderTotalCount,
    } = props;

    const handleMoreImagesClick = useCallback(() => {
        if (onMoreImagesClick) {
            onMoreImagesClick(imageSources.length - 1);
        }
    }, [onMoreImagesClick, imageSources]);

    const handleImageClick = useCallback(
        (e, index?: number | string) => {
            if (onImageClick && typeof index === 'number') {
                onImageClick(index);
            }
        },
        [onImageClick],
    );

    function renderImage(src: string, index: number): React.ReactNode {
        const isLastClickableEnabled =
            isLastClickable && index === imageSources.length - 1;

        return (
            <div
                key={index + src}
                className={cx('imageWrapper', imageWrapperClassName)}
                onClick={
                    isLastClickableEnabled ? handleMoreImagesClick : undefined
                }
            >
                <Image
                    id={index}
                    hasFitCover
                    className={cx('image')}
                    src={src}
                    onClick={handleImageClick}
                    imageAlt={imagesAlt}
                />
                {isLastClickableEnabled && (
                    <div className={cx('overlay')}>
                        {renderTotalCount(totalImageCount)}
                    </div>
                )}
            </div>
        );
    }

    return (
        <Box className={className} nowrap inline between={1}>
            {imageSources.map(renderImage)}
        </Box>
    );
};

export default GalleryRow;
