import {EHotelImageAltayStandardSize} from 'types/hotels/common/EHotelImageAltayStandardSize';
import {IHotelOrRoomImage} from 'types/hotels/hotel/IHotelImages';

import {parseImageUrlTemplate} from 'projects/hotels/utilities/prepareAndParseImages/prepareAndParseImages';

// DESKTOP
// Минимальная сумма пропорций в одном ряду
export const MIN_PROPORTIONS_SUM = 4;
// Максимальная сумма пропорций в одном ряду
export const MAX_PROPORTIONS_SUM = 8;

const DEFAULT_SIZE = {
    width: 500,
    height: 500,
};

export interface IImageUrlAndWidthResultType {
    src: string;
    width: number;
    index: number;
}

export interface IGroupedImagesWithWidth {
    items: IImageUrlAndWidthResultType[];
    totalProportion: number;
}

interface IImageUrlAndProportionResultType {
    src: string;
    proportion: number;
    index: number;
}

interface IGroupedWithProportionImages {
    items: IImageUrlAndProportionResultType[];
    totalProportion: number;
}

export const convertAndGroupImagesForRows = (
    images: IHotelOrRoomImage[],
    imageSizeName: EHotelImageAltayStandardSize,
): IGroupedImagesWithWidth[] => {
    const imagesArray: IImageUrlAndProportionResultType[] = (images || []).map(
        (image, index) => {
            const foundSize =
                image.sizes.find(({size}) => size === imageSizeName) ||
                DEFAULT_SIZE;

            return {
                src: parseImageUrlTemplate(
                    image.urlTemplate,
                    imageSizeName,
                    image.sizes,
                ),
                proportion: foundSize.width / foundSize.height,
                index,
            };
        },
    );

    const groupedImages = imagesArray.reduce<IGroupedWithProportionImages[]>(
        (acc, item) => {
            const curAcc = acc[acc.length - 1];

            // Берем последний элемент и в нем смотрим если меньше 4 элементов и сумма пропорций меньше 6.5 (плюс текущий элемент) - то добавляем картинку
            // Либо элементов 4+ и сумма пропорций меньше 4, а так же в сумме с пропорцией текущего элемента меньше 6.5
            if (
                (curAcc.items.length < 4 &&
                    curAcc.totalProportion + item.proportion <
                        MAX_PROPORTIONS_SUM) ||
                (curAcc.items.length >= 4 &&
                    curAcc.totalProportion < MIN_PROPORTIONS_SUM &&
                    curAcc.totalProportion + item.proportion <
                        MAX_PROPORTIONS_SUM)
            ) {
                curAcc.items.push(item);
                curAcc.totalProportion += item.proportion;
            } else {
                const tooSmall = acc.find(
                    searchItem =>
                        searchItem.totalProportion < MIN_PROPORTIONS_SUM,
                );

                // Проверяем есть ли ряд, у которого сумма пропорций меньше 4 и если есть и в сумме с новым элементом это будет не больше 6.5 то туда добавляем
                if (
                    tooSmall &&
                    tooSmall.totalProportion + item.proportion <
                        MAX_PROPORTIONS_SUM
                ) {
                    tooSmall.items.push(item);
                    tooSmall.totalProportion += item.proportion;
                } else {
                    acc.push({
                        totalProportion: item.proportion,
                        items: [item],
                    });
                }
            }

            return acc;
        },
        [
            {
                items: [],
                totalProportion: 0,
            },
        ],
    );

    return groupedImages.map(
        ({items, totalProportion}): IGroupedImagesWithWidth => {
            const fullWidth = items.reduce((width, item) => {
                return width + item.proportion;
            }, 0);

            return {
                items: items.map(image => ({
                    src: image.src,
                    width: (image.proportion / fullWidth) * 100,
                    index: image.index,
                })),
                totalProportion,
            };
        },
    );
};
