import React, {createRef, PureComponent, ReactNode} from 'react';
import {RouteComponentProps} from 'react-router';
import _last from 'lodash/last';

import {EHotelPageModalStatus} from 'projects/hotels/constants/hotelPage';
import {URLs} from 'constants/urls';

import {IWithClassName} from 'src/types/withClassName';
import {IHotelInfo} from 'reducers/hotels/hotelPage/hotelInfo/types';
import {EHotelsGoal} from 'utilities/metrika/types/goals/hotels';
import {EFavoritesGoal} from 'utilities/metrika/types/goals/favorites';
import {IGeoRegion} from 'types/hotels/hotel/IGeoRegion';

import {IDevice} from 'reducers/common/commonReducerTypes';

import {prepareQaAttributes} from 'utilities/qaAttributes/qaAttributes';
import {hotelsURLs} from 'projects/hotels/utilities/urls';
import {insertJSXIntoKey} from 'utilities/tanker/insertJSXIntoKey';
import debounceById from 'utilities/functions/debounceById';
import {deviceModMobile, deviceMods} from 'utilities/stylesUtils';
import {reachGoal} from 'utilities/metrika';

import * as i18nFavoritesBlock from 'i18n/account-Favorites';

import {IHotelPageCardContainer} from 'projects/hotels/pages/HotelPage/components/HotelPageCard/HotelPageCardContainer';
import HotelPageCardMainTab from 'projects/hotels/pages/HotelPage/components/HotelPageCardMainTab/HotelPageCardMainTab';
import HotelPageCardHeaderDesktop from 'projects/hotels/pages/HotelPage/components/HotelPageCardHeaderDesktop/HotelPageCardHeaderDesktop';
import Modal, {EModalAnimationType, IModalProps} from 'components/Modal/Modal';
import MobileHotelPageRoomDetailedInfo from 'projects/hotels/pages/HotelPage/components/HotelPageAllAmenities/blocks/MobileHotelPageRoomDetailedInfo/MobileHotelPageRoomDetailedInfo';
import HotelPageAllAmenities from 'projects/hotels/pages/HotelPage/components/HotelPageAllAmenities/HotelPageAllAmenities';
import HotelPageCardMap from 'projects/hotels/pages/HotelPage/components/HotelPageCardMap/HotelPageCardMap';
import BottomSheet, {
    IBottomSheetProps,
} from 'components/BottomSheet/BottomSheet';
import LinkButton from 'components/LinkButton/LinkButton';
import Snackbar from 'projects/hotels/components/Snackbar/Snackbar';
import NoMoreFavoritesModal from 'projects/favorites/components/NoMoreFavoritesModal/NoMoreFavoritesModal';
import CloseForCircleBgIcon from 'icons/24/CloseForCircleBg';
import Button from 'components/Button/Button';
import HotelPageCardBanner from 'projects/hotels/pages/HotelPage/components/HotelPageCardBanner/HotelPageCardBanner';
import BackButton from 'projects/hotels/pages/HotelPage/components/BackButton/BackButton';

import cx from './HotelPageCard.scss';

/* Component Types */
interface IHotelPageCardProps extends IHotelPageCardContainer, IWithClassName {
    deviceType: IDevice;
    location: TReactRouterLocation;
    hotelInfo: IHotelInfo;
    isLoadingOffers: boolean;
    isHeadless: boolean;
}

interface IHotelPageCardState {
    modalStatus?: EHotelPageModalStatus;
    canPrerender?: boolean;
    roomId?: string;
}

type TReactRouterLocation = RouteComponentProps['location'];

/* Constants */
const HOTEL_PAGE_QA = 'hotels-hotelPage';
const MAP_TAB_QA = 'hotelMapTab';
const SET_ON_FAVORITES_CLICK_DEBOUNCE = 2000;

class HotelPageCard extends PureComponent<
    IHotelPageCardProps,
    IHotelPageCardState
> {
    reviewsRef = createRef<HTMLDivElement>();

    private debouncedSearchReactionRequest = debounceById(
        (permalink: string) => {
            this.props.changeHotelsIsFavoriteAction({permalink});
        },
        SET_ON_FAVORITES_CLICK_DEBOUNCE,
    );

    constructor(props: IHotelPageCardProps) {
        super(props);

        this.state = {
            modalStatus: EHotelPageModalStatus.closed,
        };
    }

    state: IHotelPageCardState;

    componentDidMount(): void {
        this.setState({canPrerender: true});
    }

    /* Handlers */

    private handleReviewsClick = (): void => {
        if (this.reviewsRef.current) {
            const {deviceType} = this.props;
            const headerHeight = deviceType.isMobile ? 0 : 80;
            const y =
                this.reviewsRef.current.getBoundingClientRect().top +
                window.pageYOffset -
                headerHeight;

            window.scrollTo({top: y, behavior: 'smooth'});

            reachGoal(EHotelsGoal.HOTELS_HOTEL_PAGE_REVIEWS_COUNT_CLICK);
        }
    };

    private handleStaticMapClick = (): void => {
        reachGoal(EHotelsGoal.HOTELS_HOTEL_PAGE_STATIC_MAP_CLICK);

        this.setState({
            modalStatus: EHotelPageModalStatus.map,
        });
    };

    private handleFavoritesButtonClick = (): void => {
        const {
            hotelInfo,
            removeFavoriteHotel,
            addFavoriteHotel,
            setSnackbarInfo,
        } = this.props;

        const {
            hotel: {permalink, isFavorite},
        } = hotelInfo;

        if (isFavorite) {
            removeFavoriteHotel({permalink});
            setSnackbarInfo({
                lastAction: `${permalink}-remove`,
                hasCancellation: true,
                message: i18nFavoritesBlock.removeFavoriteHotel(),
                cancelAction: 'addFavoriteHotelOnHotelPage',
                cancelActionData: {requestParams: {permalink}},
                page: 'hotel',
            });
        } else {
            addFavoriteHotel({
                requestParams: {permalink},
                message: insertJSXIntoKey(i18nFavoritesBlock.addFavoriteHotel)({
                    favorites: (
                        <LinkButton href={URLs.favorites} theme="brand">
                            {i18nFavoritesBlock.favorites()}
                        </LinkButton>
                    ),
                }),
            });
        }

        reachGoal(EFavoritesGoal.FAVORITES_CLICK);

        this.debouncedSearchReactionRequest(permalink, permalink);
    };

    private handleOffersButtonClick = (): void => {
        this.setState({roomId: undefined});
    };

    private handleOfferWatchButtonClick = (roomId: string): void => {
        this.setState({
            roomId,
            modalStatus: EHotelPageModalStatus.allAmenities,
        });
    };

    /* Actions */

    private closeModalOrBottomSheet = (): void => {
        this.setState({modalStatus: EHotelPageModalStatus.closed});
    };

    private getGeoRegion = (): IGeoRegion | undefined => {
        const {
            hotelInfo: {
                breadcrumbs: {geoRegions},
            },
        } = this.props;

        return _last(geoRegions);
    };

    private getGeoId = (): number | undefined => {
        const region = this.getGeoRegion();

        return region?.geoId;
    };

    private getBackButtonLink = (): string | undefined => {
        const {location} = this.props;

        const geoId = this.getGeoId();

        return geoId
            ? hotelsURLs.getHotelsSearchUrl(location, geoId)
            : undefined;
    };

    /* Render */
    renderCardHeaderMobile(): ReactNode {
        const {hotelInfo, deviceType} = this.props;

        const {
            breadcrumbs: {geoRegions},
        } = hotelInfo;

        const region = _last(geoRegions);

        return (
            <>
                <BackButton
                    backGeoRegionLink={this.getBackButtonLink()}
                    backGeoRegion={region}
                    {...prepareQaAttributes(HOTEL_PAGE_QA)}
                />

                <HotelPageCardBanner
                    className={cx(
                        'banner',
                        deviceModMobile('banner', deviceType),
                    )}
                    hotelInfo={hotelInfo}
                    {...prepareQaAttributes(HOTEL_PAGE_QA)}
                />
            </>
        );
    }
    renderCardHeaderDesktop(): ReactNode {
        const {hotelInfo, deviceType} = this.props;
        const {
            hotel,
            ratingsInfo,
            breadcrumbs: {geoRegions},
            searchParams,
        } = hotelInfo;

        const region = _last(geoRegions);

        return (
            <>
                <HotelPageCardBanner
                    className={cx(
                        'banner',
                        deviceModMobile('banner', deviceType),
                    )}
                    hotelInfo={hotelInfo}
                    {...prepareQaAttributes(HOTEL_PAGE_QA)}
                />

                <HotelPageCardHeaderDesktop
                    className={cx('header')}
                    hotel={hotel}
                    ratingsInfo={ratingsInfo}
                    withoutDates={!searchParams}
                    backGeoRegionLink={this.getBackButtonLink()}
                    backGeoRegion={region}
                    onHotelAddressClick={this.handleStaticMapClick}
                    deviceType={deviceType}
                    onFavoritesClick={this.handleFavoritesButtonClick}
                    onReviewsClick={this.handleReviewsClick}
                    {...prepareQaAttributes(HOTEL_PAGE_QA)}
                />
            </>
        );
    }

    renderHotelCardMainTab(): ReactNode {
        const {
            deviceType,
            hotelInfo,
            isLoadingOffers,
            getHotelImages,
            hotelImages,
            experiments: {roomsMatching},
        } = this.props;

        return (
            <HotelPageCardMainTab
                deviceType={deviceType}
                isLoadingOffers={isLoadingOffers}
                onOffersButtonClick={this.handleOffersButtonClick}
                onFavoritesClick={this.handleFavoritesButtonClick}
                handleOfferWatchButtonClick={this.handleOfferWatchButtonClick}
                onReviewsButtonClick={this.handleReviewsClick}
                onStaticMapClick={this.handleStaticMapClick}
                backButtonLink={this.getBackButtonLink()}
                hotelInfo={hotelInfo}
                onHotelAddressClick={this.handleStaticMapClick}
                getHotelImages={getHotelImages}
                hotelImages={hotelImages}
                reviewsRef={this.reviewsRef}
                withRoomsMatching={roomsMatching}
                region={this.getGeoRegion()}
                {...prepareQaAttributes(HOTEL_PAGE_QA)}
            />
        );
    }

    renderContent(): ReactNode {
        const {
            isHeadless,
            changeIsOpenForNoMoreFavoritesModal,
            deviceType,
            noMoreFavoritesModalIsVisible,
        } = this.props;

        if (isHeadless) {
            return this.renderHotelCardMainTab();
        }

        const onCloseModalClick = (): void => {
            changeIsOpenForNoMoreFavoritesModal(false);
        };

        const header = deviceType.isMobile
            ? this.renderCardHeaderMobile()
            : this.renderCardHeaderDesktop();

        return (
            <>
                {header}
                {this.renderHotelCardMainTab()}
                <Snackbar deviceType={deviceType} currentPage="hotel" />
                <NoMoreFavoritesModal
                    onClose={onCloseModalClick}
                    isVisible={Boolean(noMoreFavoritesModalIsVisible)}
                />
            </>
        );
    }

    wrapModalContentIfNeed(content: ReactNode): ReactNode {
        const {modalStatus} = this.state;

        if (modalStatus === EHotelPageModalStatus.map) {
            return content;
        }

        return <Modal.Content>{content}</Modal.Content>;
    }

    renderModalOrBottomSheet(): ReactNode {
        const {deviceType} = this.props;
        const {isMobile} = deviceType;
        const content = this.renderModalOrBottomSheetContent();

        if (isMobile) {
            return (
                <BottomSheet {...this.getBottomSheetProps()}>
                    {content}
                </BottomSheet>
            );
        }

        return (
            <Modal {...this.getModalProps()}>
                {this.wrapModalContentIfNeed(content)}
            </Modal>
        );
    }

    getModalProps(): IModalProps {
        const {modalStatus} = this.state;
        const commonProps: Partial<IModalProps> = {
            onClose: this.closeModalOrBottomSheet,
            animation: EModalAnimationType.NONE,
        };

        switch (modalStatus) {
            case EHotelPageModalStatus.allAmenities: {
                return {
                    ...commonProps,
                    isVisible:
                        modalStatus === EHotelPageModalStatus.allAmenities,
                    className: cx('modal_allAmenities'),
                    containerClassName: cx('modal_allAmenitiesContainer'),
                };
            }

            case EHotelPageModalStatus.map: {
                return {
                    ...commonProps,
                    isVisible: modalStatus === EHotelPageModalStatus.map,
                    containerClassName: cx('modalContainerMap'),
                    renderCloseButton: (close): React.ReactNode => (
                        <Button
                            className={cx('closeMapButton')}
                            shape="circle"
                            theme="raised"
                            onClick={close}
                        >
                            <CloseForCircleBgIcon />
                        </Button>
                    ),
                };
            }

            default: {
                return {
                    ...commonProps,
                    isVisible: false,
                };
            }
        }
    }

    getBottomSheetProps(): IBottomSheetProps {
        const {modalStatus} = this.state;
        const commonProps = {
            onClose: this.closeModalOrBottomSheet,
        };

        switch (modalStatus) {
            case EHotelPageModalStatus.allAmenities: {
                return {
                    ...commonProps,
                    isOpened:
                        modalStatus === EHotelPageModalStatus.allAmenities,
                };
            }

            case EHotelPageModalStatus.map: {
                return {
                    ...commonProps,
                    isOpened: modalStatus === EHotelPageModalStatus.map,
                    contentClassName: cx('bottomSheet_map'),
                    dragDisabledContent: true,
                };
            }

            default: {
                return {
                    ...commonProps,
                    isOpened: false,
                };
            }
        }
    }

    renderModalOrBottomSheetContent(): ReactNode {
        const {
            deviceType,
            hotelInfo: {
                hotel,
                offersInfo: {rooms},
            },
        } = this.props;
        const {isMobile} = deviceType;
        const {modalStatus, roomId} = this.state;

        switch (modalStatus) {
            case EHotelPageModalStatus.allAmenities: {
                if (roomId === undefined) {
                    return null;
                }

                const room = rooms?.find(
                    currentRoom => currentRoom.id === roomId,
                );

                if (!room) {
                    return null;
                }

                if (isMobile) {
                    return <MobileHotelPageRoomDetailedInfo room={room} />;
                }

                return <HotelPageAllAmenities room={room} />;
            }

            case EHotelPageModalStatus.map: {
                return (
                    <HotelPageCardMap
                        className={cx(deviceMods('map', deviceType))}
                        hotel={hotel}
                        canRenderAddress={isMobile}
                        deviceType={deviceType}
                        {...prepareQaAttributes(MAP_TAB_QA)}
                    />
                );
            }

            default: {
                return null;
            }
        }
    }

    render(): ReactNode {
        const {className, deviceType} = this.props;

        return (
            <div
                className={cx(deviceMods('root', deviceType), className)}
                {...prepareQaAttributes('hotelCard')}
            >
                {this.renderContent()}
                {this.renderModalOrBottomSheet()}
            </div>
        );
    }
}

export default HotelPageCard;
