import React, {ReactElement, ReactNode, useCallback} from 'react';
import noop from 'lodash/noop';

import {IHotel} from 'types/hotels/hotel/IHotel';
import {IWithDeviceType} from 'types/withDeviceType';
import {IWithClassName} from 'src/types/withClassName';
import {IRatingsInfo} from 'server/api/HotelSearchAPI/types/IHotelInfo';
import EPopupDirection from 'components/Popup/types/EPopupDirection';

import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {deviceMods} from 'utilities/stylesUtils';

import HotelAddress from 'projects/hotels/components/HotelAddress/HotelAddress';
import TransportAccessibility from 'projects/hotels/components/TransportAccessibility/TransportAccessibility';
import HotelRatingWithReviewLink from 'projects/hotels/components/HotelRatingWithReviewLink/HotelRatingWithReviewLink';
import Dropdown, {IDropdownSwitcherParams} from 'components/Dropdown/Dropdown';
import HotelFeatures from 'projects/hotels/components/HotelReviews/components/HotelFeatures/HotelFeatures';
import HotelTitle from 'projects/hotels/pages/HotelPage/components/HotelTitle/HotelTitle';
import Location from 'icons/16/Location';
import HotelGeoFeature from 'projects/hotels/components/HotelGeoFeature/HotelGeoFeature';
import Box from 'components/Box/Box';
import BackButton, {
    IBackButtonProps,
} from 'projects/hotels/pages/HotelPage/components/BackButton/BackButton';

import cx from './HotelPageCardHeaderDesktop.scss';

export interface IHotelPageCardHeaderProps
    extends IWithClassName,
        IWithDeviceType,
        IWithQaAttributes {
    hotel: IHotel;
    ratingsInfo: IRatingsInfo;
    withoutDates: boolean;
    backGeoRegionLink?: IBackButtonProps['backGeoRegionLink'];
    backGeoRegion?: IBackButtonProps['backGeoRegion'];
    onTotalReviewLinkClick?: () => void;
    onReviewsClick: () => void;
    onHotelAddressClick?: () => void;
    onFavoritesClick: () => void;
}

const HotelPageCardHeaderDesktop: React.FC<IHotelPageCardHeaderProps> =
    props => {
        const {
            className,
            deviceType,
            backGeoRegionLink,
            backGeoRegion,
            hotel,
            ratingsInfo,
            onReviewsClick = noop,
            onHotelAddressClick = noop,
            onFavoritesClick,
        } = props;

        const renderHotelName = (): ReactNode => {
            const {name, stars, isFavorite} = hotel;

            return (
                <HotelTitle
                    name={name}
                    stars={stars}
                    isFavorite={isFavorite}
                    handleFavoriteClick={onFavoritesClick}
                    deviceType={deviceType}
                />
            );
        };

        const renderHotelAddress = (): ReactNode => {
            const {address} = hotel;

            return (
                <HotelAddress
                    className={cx('address')}
                    text={address}
                    onClick={onHotelAddressClick}
                    type="separate-link"
                    icon={Location}
                    multiLine
                />
            );
        };

        const renderGeoFeature = (): ReactNode => {
            const {geoFeature} = hotel;

            return (
                <HotelGeoFeature
                    geoFeature={geoFeature}
                    {...prepareQaAttributes({
                        current: 'geoFeature',
                        parent: props,
                    })}
                />
            );
        };

        const renderTransportAccessibility = (): ReactNode => {
            const {nearestStations} = hotel;

            return (
                nearestStations?.[0] && (
                    <TransportAccessibility
                        textClassName={cx('transportAccessibility_text')}
                        bigIcon
                        station={nearestStations[0]}
                        {...prepareQaAttributes({
                            current: 'transportAccessibility',
                            parent: props,
                        })}
                    />
                )
            );
        };

        const renderHotelRating = useCallback((): ReactElement => {
            const {rating, totalTextReviewCount} = hotel;

            return (
                <HotelRatingWithReviewLink
                    className={cx('hotelRating')}
                    rating={rating}
                    totalTextReviewCount={totalTextReviewCount}
                    isMobile={false}
                />
            );
        }, [hotel]);

        const renderPositiveReview = useCallback((): ReactNode => {
            if (!ratingsInfo?.teaser) {
                return null;
            }

            return (
                <div
                    className={cx('positiveReview')}
                    {...prepareQaAttributes('positiveReview')}
                >
                    {ratingsInfo.teaser}
                </div>
            );
        }, [ratingsInfo]);

        const renderDesktopRightColumnSwitcher = useCallback(
            ({
                input: {ref},
                meta: {showPopup, hidePopup},
            }: IDropdownSwitcherParams<unknown>): React.ReactElement => {
                const {featureRatings} = ratingsInfo;
                const canRenderFeatures = featureRatings?.length > 0;
                const handlers = canRenderFeatures
                    ? {
                          onMouseEnter: showPopup,
                          onMouseLeave: hidePopup,
                      }
                    : {};

                return (
                    <div
                        ref={ref}
                        className={cx('rightColumn')}
                        onClick={onReviewsClick}
                        {...handlers}
                    >
                        {renderHotelRating()}
                        {renderPositiveReview()}
                    </div>
                );
            },
            [
                ratingsInfo,
                onReviewsClick,
                renderHotelRating,
                renderPositiveReview,
            ],
        );

        const renderDesktopRightColumnPopup =
            useCallback((): React.ReactNode => {
                return <HotelFeatures />;
            }, []);

        const renderDesktopRightColumn = (): React.ReactElement => {
            return (
                <Dropdown
                    renderSwitcher={renderDesktopRightColumnSwitcher}
                    renderPopup={renderDesktopRightColumnPopup}
                    popupProps={{
                        directions: [EPopupDirection.BOTTOM_RIGHT],
                        className: cx('rightColumn_popup'),
                    }}
                />
            );
        };

        return (
            <>
                <BackButton
                    backGeoRegion={backGeoRegion}
                    backGeoRegionLink={backGeoRegionLink}
                    {...prepareQaAttributes(props)}
                />
                <div className={cx(deviceMods('root', deviceType), className)}>
                    <div className={cx('leftColumn')}>
                        {renderHotelName()}
                        <Box between={1}>
                            {renderHotelAddress()}
                            <Box between={4} inline>
                                {renderGeoFeature()}
                                {renderTransportAccessibility()}
                            </Box>
                        </Box>
                    </div>
                    {renderDesktopRightColumn()}
                </div>
            </>
        );
    };

export default React.memo(HotelPageCardHeaderDesktop);
