import React, {FC, useCallback, useRef} from 'react';

import {IHotelInfo} from 'server/api/HotelSearchAPI/types/IHotelInfo';
import {EHotelsGoal} from 'utilities/metrika/types/goals/hotels';
import {IWithClassName} from 'types/withClassName';

import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {useReachGoal} from 'utilities/metrika/useReachGoal';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import * as i18nBlock from 'i18n/hotels-HotelPage';

import Anchor from 'components/Anchor/Anchor';
import HotelPageAmenities from 'projects/hotels/pages/HotelPage/components/HotelPageCardMainTab/blocks/HotelPageAmenities/HotelPageAmenities';
import Amenities from 'projects/hotels/components/Amenities/Amenities';
import Heading from 'components/Heading/Heading';

import cx from './HotelPageAboutHotelInfo.scss';

interface IHotelPageAboutHotelInfoProps
    extends IWithClassName,
        IWithQaAttributes {
    hotelInfo: IHotelInfo;
}

const HotelPageAboutHotelInfo: FC<IHotelPageAboutHotelInfoProps> = ({
    hotelInfo,
    className,
    ...qaProps
}) => {
    const deviceType = useDeviceType();

    const {isMobile, isDesktop} = deviceType;
    const {
        hotel: {name, mainAmenities, amenityGroups},
        hotelDescription,
    } = hotelInfo;

    const hasAmenities =
        (mainAmenities && mainAmenities.length) ||
        (amenityGroups && amenityGroups.length);

    const hasDescription = Boolean(hotelDescription?.text);

    const hasAdditionalAmenities = Boolean(
        amenityGroups && amenityGroups.length,
    );

    const showAmenitiesText = isDesktop
        ? i18nBlock.showAmenities()
        : i18nBlock.toggleAmenitiesButton();

    const aboutHotelInfoRef = useRef<HTMLDivElement>(null);
    const handleShowAmenities = useReachGoal(
        EHotelsGoal.HOTEL_PAGE_MAIN_TAB_SHOW_AMENITIES,
    );

    const scrollToAboutHotelInfo = useCallback((): void => {
        aboutHotelInfoRef.current?.scrollIntoView();
    }, []);

    if (!hasAmenities && !hasDescription) {
        return null;
    }

    return (
        <div
            className={cx('root', className)}
            {...prepareQaAttributes(qaProps)}
        >
            {!isMobile && (
                <>
                    <Anchor anchorRef={aboutHotelInfoRef} />
                    <Heading level={2} className={cx('heading')}>
                        {i18nBlock.aboutHotelLabel()}
                    </Heading>
                </>
            )}
            {hasDescription && (
                <div className={cx('description')}>{hotelDescription.text}</div>
            )}

            {hasAmenities && (
                <HotelPageAmenities
                    hasAdditionalAmenities={hasAdditionalAmenities}
                    backButtonText={name}
                    onShowAmenities={handleShowAmenities}
                    mobilePopupTitle={i18nBlock.amenitiesTitle()}
                    showAmenitiesText={showAmenitiesText}
                    hideAmenitiesText={i18nBlock.hideAmenities()}
                    onHideAmenities={scrollToAboutHotelInfo}
                    renderDesktopAmenities={(
                        isAll: boolean,
                    ): React.ReactNode => (
                        <Amenities
                            mainAmenities={mainAmenities}
                            amenityGroups={amenityGroups}
                            isAllAmenities={isAll}
                            deviceType={deviceType}
                        />
                    )}
                    renderMobileMainAmenities={(): React.ReactNode => (
                        <Amenities
                            mainAmenities={mainAmenities}
                            amenityGroups={amenityGroups}
                            isAllAmenities={false}
                            deviceType={deviceType}
                        />
                    )}
                    renderMobileModalAmenities={(): React.ReactNode => (
                        <Amenities
                            mainAmenities={mainAmenities}
                            amenityGroups={amenityGroups}
                            isAllAmenities={true}
                            deviceType={deviceType}
                        />
                    )}
                    deviceType={deviceType}
                    {...prepareQaAttributes(qaProps)}
                />
            )}
        </div>
    );
};

export default HotelPageAboutHotelInfo;
