import {PureComponent, ReactNode} from 'react';

import {IDevice} from 'reducers/common/commonReducerTypes';

import {deviceMods} from 'utilities/stylesUtils';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import LinkButton from 'components/LinkButton/LinkButton';
import Heading from 'components/Heading/Heading';
import TextWithIcon from 'components/TextWithIcon/TextWithIcon';
import ArrowBottomIcon from 'icons/16/ArrowBottom';
import ArrowTopIcon from 'icons/16/ArrowTop';
import ArrowRightIcon from 'icons/16/ArrowRight';
import BottomSheet from 'components/BottomSheet/BottomSheet';

import cx from './HotelPageAmenities.scss';

interface IHotelPageAmenitiesProps extends IWithQaAttributes {
    renderMobileMainAmenities: () => ReactNode;
    renderMobileModalAmenities: () => ReactNode;
    renderDesktopAmenities: (isAllAmenities: boolean) => ReactNode;
    deviceType: IDevice;
    className?: string;
    backButtonText?: string;
    hasAdditionalAmenities: boolean;
    showAmenitiesText: string;
    hideAmenitiesText: string;
    mobilePopupTitle: string;
    mobilePopupTitleClassName?: string;
    onShowAmenities?: () => void;
    onHideAmenities?: () => void;
}

interface IHotelPageAmenitiesState {
    isAllAmenities: boolean;
    canRenderModalAmenities: boolean;
}

class HotelPageAmenities extends PureComponent<
    IHotelPageAmenitiesProps,
    IHotelPageAmenitiesState
> {
    state: IHotelPageAmenitiesState = {
        isAllAmenities: false,
        canRenderModalAmenities: false,
    };

    /* Handlers */

    private handleToggleAmenities = (): void => {
        const {onShowAmenities, onHideAmenities} = this.props;

        this.setState(({isAllAmenities}) => ({
            isAllAmenities: !isAllAmenities,
        }));

        if (!this.state.isAllAmenities && onShowAmenities) {
            onShowAmenities();
        }

        if (this.state.isAllAmenities && onHideAmenities) {
            onHideAmenities();
        }
    };

    private handleShowModalAmenities = (): void => {
        const {onShowAmenities} = this.props;

        this.setState({canRenderModalAmenities: true});

        if (onShowAmenities) {
            onShowAmenities();
        }
    };

    private handleModalAmenitiesClose = (): void => {
        this.setState({canRenderModalAmenities: false});
    };

    /* Render */

    private renderAmenitiesModal(): ReactNode {
        const {
            renderMobileModalAmenities,
            mobilePopupTitle,
            mobilePopupTitleClassName,
        } = this.props;
        const {canRenderModalAmenities} = this.state;

        return (
            <BottomSheet
                isOpened={canRenderModalAmenities}
                onClose={this.handleModalAmenitiesClose}
            >
                <div>
                    <Heading
                        level={2}
                        className={cx(
                            'modalAmenitiesTitle',
                            mobilePopupTitleClassName,
                        )}
                    >
                        {mobilePopupTitle}
                    </Heading>
                    {renderMobileModalAmenities()}
                </div>
            </BottomSheet>
        );
    }

    private renderToggleAmenitiesButton(): ReactNode {
        const {hasAdditionalAmenities, showAmenitiesText, hideAmenitiesText} =
            this.props;

        if (!hasAdditionalAmenities) {
            return null;
        }

        const {isAllAmenities} = this.state;

        const Icon = isAllAmenities ? ArrowTopIcon : ArrowBottomIcon;
        const toggleAmenitiesText = isAllAmenities
            ? hideAmenitiesText
            : showAmenitiesText;

        return (
            <LinkButton
                theme="normal"
                className={cx('toggleAmenities', {
                    toggleAmenities_type_all: isAllAmenities,
                })}
                onClick={this.handleToggleAmenities}
                {...prepareQaAttributes({
                    parent: this.props,
                    current: 'toggleAmenities',
                })}
            >
                <TextWithIcon
                    iconRight={Icon}
                    iconRightClassName={cx('arrowIcon')}
                    text={toggleAmenitiesText}
                />
            </LinkButton>
        );
    }

    private renderShowModalAmenitiesButton(): ReactNode {
        const {deviceType, showAmenitiesText, hasAdditionalAmenities} =
            this.props;

        if (!hasAdditionalAmenities) {
            return null;
        }

        const iconsProps = deviceType.isDesktop
            ? {
                  iconRight: ArrowRightIcon,
                  iconRightClassName: cx('arrowIcon'),
              }
            : {};

        return (
            <LinkButton
                theme="normal"
                className={cx(
                    'toggleAmenities',
                    deviceMods('toggleAmenities', deviceType),
                )}
                onClick={this.handleShowModalAmenities}
                {...prepareQaAttributes({
                    parent: this.props,
                    current: 'toggleAmenities',
                })}
            >
                <TextWithIcon {...iconsProps} text={showAmenitiesText} />
            </LinkButton>
        );
    }

    private renderMobile(): ReactNode {
        const {renderMobileMainAmenities} = this.props;

        return (
            <>
                {renderMobileMainAmenities()}
                {this.renderShowModalAmenitiesButton()}
                {this.renderAmenitiesModal()}
            </>
        );
    }

    private renderDesktop(): ReactNode {
        const {renderDesktopAmenities} = this.props;
        const {isAllAmenities} = this.state;

        return (
            <>
                {renderDesktopAmenities(isAllAmenities)}
                {this.renderToggleAmenitiesButton()}
            </>
        );
    }

    render(): ReactNode {
        const {
            deviceType: {isMobile},
        } = this.props;

        return isMobile ? this.renderMobile() : this.renderDesktop();
    }
}

export default HotelPageAmenities;
