import React, {useState, FC, useEffect, useCallback, useMemo} from 'react';

import {EHotelsGoal} from 'utilities/metrika/types/goals/hotels';
import {IHotelInfo} from 'server/api/HotelSearchAPI/types/IHotelInfo';
import {IWithClassName} from 'types/withClassName';

import {GetHotelImagesActionType} from 'reducers/hotels/hotelPage/hotelImages/actions';

import {THotelImagesSelectorType} from 'selectors/hotels/hotel/imagesTab/getHotelImages';

import {reachGoal} from 'utilities/metrika';
import {prepareQaAttributes} from 'utilities/qaAttributes/qaAttributes';
import {useDeviceType} from 'utilities/hooks/useDeviceType';

import HotelPageMobileMainImage from 'projects/hotels/pages/HotelPage/components/HotelPageCardMainTab/blocks/HotelPageMobileMainImage/HotelPageMobileMainImage';
import Separator from 'components/Separator/Separator';
import Gallery from 'projects/hotels/pages/HotelPage/components/Gallery/Gallery';
import HotelImages from 'projects/hotels/pages/HotelPage/components/HotelImages/HotelImagesContainer';

import cx from './HotelPageCardGallery.scss';

export interface IHotelPageCardGalleryProps extends IWithClassName {
    hotelInfo: IHotelInfo;
    hotelImages: THotelImagesSelectorType;
    getHotelImages: GetHotelImagesActionType;
}

const ITEMS_TO_LOAD_MOBILE = 10;
const ITEMS_TO_LOAD_DESKTOP = 16;
const MAX_ITEMS_LEFT = 20;
const DEFAULT_IMAGE_LIMIT = 20;

const HotelPageCardGallery: FC<IHotelPageCardGalleryProps> = ({
    hotelImages,
    hotelInfo,
    getHotelImages,
    className,
}) => {
    const deviceType = useDeviceType();
    const {isMobile} = deviceType;

    const [isImagesLoaded, setIsImagesLoaded] = useState(false);
    const [modalImagesViewerState, setModalImagesViewerState] = useState({
        isModalImagesViewerVisible: false,
        viewerImageIndex: 0,
    });

    useEffect(() => {
        const {data, isLoading, isSuccess} = hotelImages;

        if (
            !isImagesLoaded &&
            !isSuccess &&
            !isLoading &&
            data.images.length === 0
        ) {
            setIsImagesLoaded(true);

            getHotelImages({
                imageOffset: hotelImages.data.images.length,
                imageLimit: DEFAULT_IMAGE_LIMIT,
            });
        }
    }, [getHotelImages, hotelImages, isImagesLoaded]);

    const handleImagesClick = useCallback(
        (index: number) => {
            reachGoal(EHotelsGoal.HOTEL_PAGE_IMAGES_TAB_IMAGE_CLICK);

            setModalImagesViewerState({
                isModalImagesViewerVisible: true,
                viewerImageIndex: index,
            });
        },
        [setModalImagesViewerState],
    );

    const loadImagesByCount = useCallback(
        (itemsCount: number) => () => {
            const {isLoading, data} = hotelImages;

            if (!isLoading && data.images.length < data.totalImageCount) {
                getHotelImages({
                    imageOffset: data.images.length,
                    imageLimit: itemsCount,
                });
            }
        },
        [getHotelImages, hotelImages],
    );

    const getLoadImageCount = useCallback((): number => {
        const {
            data: {images, totalImageCount},
        } = hotelImages;

        if (isMobile) {
            return Math.min(
                ITEMS_TO_LOAD_MOBILE,
                totalImageCount - images.length,
            );
        }

        if (totalImageCount - images.length < MAX_ITEMS_LEFT) {
            return totalImageCount - images.length;
        }

        return ITEMS_TO_LOAD_DESKTOP;
    }, [hotelImages, isMobile]);

    const handleImagesCloseClick = useCallback((): void => {
        setModalImagesViewerState({
            isModalImagesViewerVisible: false,
            viewerImageIndex: 0,
        });
    }, [setModalImagesViewerState]);

    const mobileGallery = useMemo((): React.ReactNode => {
        const {
            hotel: {images, name},
        } = hotelInfo;

        const loadImageCount = getLoadImageCount();

        if (!images.length) {
            return null;
        }

        return (
            <HotelPageMobileMainImage
                className={className}
                images={images}
                imagesAlt={name}
                hotelImages={hotelImages}
                onImagesClick={handleImagesClick}
                loadMoreImages={loadImagesByCount(loadImageCount)}
                {...prepareQaAttributes('hotelPageGallery')}
            />
        );
    }, [
        className,
        hotelInfo,
        getLoadImageCount,
        hotelImages,
        handleImagesClick,
        loadImagesByCount,
    ]);

    const desktopGallery = useMemo((): React.ReactNode => {
        const {
            hotel: {images, name, totalImageCount},
        } = hotelInfo;

        if (!totalImageCount) {
            return <Separator className={cx('separator')} />;
        }

        return (
            <Gallery
                className={className}
                images={images}
                imagesAlt={name}
                totalImageCount={totalImageCount}
                onMoreImagesClick={handleImagesClick}
            />
        );
    }, [className, hotelInfo, handleImagesClick]);

    const modalImagesViewer = useMemo((): React.ReactNode => {
        const {hotel} = hotelInfo;
        const {isModalImagesViewerVisible} = modalImagesViewerState;

        return (
            <HotelImages
                hotel={hotel}
                isVisible={isModalImagesViewerVisible}
                onCloseClick={handleImagesCloseClick}
            />
        );
    }, [handleImagesCloseClick, hotelInfo, modalImagesViewerState]);

    return (
        <>
            {isMobile ? mobileGallery : desktopGallery}
            {modalImagesViewer}
        </>
    );
};

export default HotelPageCardGallery;
