import React, {FC, useMemo} from 'react';

import {
    RIGHT_COLUMN_OFFSET,
    RIGHT_COLUMN_WIDTH,
} from 'projects/hotels/constants/hotelPage';

import {IHotelInfo} from 'server/api/HotelSearchAPI/types/IHotelInfo';

import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {prepareQaAttributes} from 'utilities/qaAttributes/qaAttributes';
import {useExperiments} from 'utilities/hooks/useExperiments';

import {TwoColumnLayout} from 'components/Layouts/TwoColumnLayout/TwoColumnLayout';
import Separator from 'components/Separator/Separator';
import HotelPageGeoInfo from 'projects/hotels/pages/HotelPage/components/HotelPageGeoInfo/HotelPageGeoInfo';
import HotelPageCardGallery, {
    IHotelPageCardGalleryProps,
} from 'projects/hotels/pages/HotelPage/components/HotelPageCardMainTab/blocks/HotelPageCardGallery/HotelPageCardGallery';
import HotelFeatures from 'projects/hotels/components/HotelReviews/components/HotelFeatures/HotelFeatures';
import HotelPageAboutHotelInfo from 'projects/hotels/pages/HotelPage/components/HotelPageCardMainTab/blocks/HotelPageAboutHotelInfo/HotelPageAboutHotelInfo';
import HotelPageSimilarHotels from 'projects/hotels/pages/HotelPage/components/HotelPageCardMainTab/blocks/HotelPageSimilarHotels/HotelPageSimilarHotels';
import HotelReviews from 'projects/hotels/components/HotelReviews/HotelReviews';
import HotelPageOffersInfo from 'projects/hotels/pages/HotelPage/components/HotelPageCardMainTab/blocks/HotelPageOffersInfo/HotelPageOffersInfo';

import cx from './HotelPageCardMainTabDesktop.scss';

interface IHotelPageCardMainTabDesktopProps {
    hotelInfo: IHotelInfo;

    onMapClick: () => void;
    onOffersButtonClick: () => void;
    onOfferWatchButtonClick: (roomId: string) => void;

    hotelImages: IHotelPageCardGalleryProps['hotelImages'];
    getHotelImages: IHotelPageCardGalleryProps['getHotelImages'];

    reviewsRef: React.RefObject<HTMLDivElement>;

    withRoomsMatching?: boolean;
    isLoadingOffers: boolean;
    backButtonLink?: string;
}

const HotelPageCardMainTabDesktop: FC<IHotelPageCardMainTabDesktopProps> = ({
    hotelInfo,
    hotelImages,
    getHotelImages,
    onMapClick,
    reviewsRef,
    isLoadingOffers,
    withRoomsMatching,
    backButtonLink,
    onOffersButtonClick,
    onOfferWatchButtonClick,
}) => {
    const deviceType = useDeviceType();

    const hasReviews = Boolean(hotelInfo.reviewsInfo);

    const {hotelsFeaturesDown} = useExperiments();

    const aboutHotelNode = useMemo(
        () => (
            <HotelPageAboutHotelInfo
                className={cx('gapTop')}
                hotelInfo={hotelInfo}
                {...prepareQaAttributes('aboutHotel')}
            />
        ),
        [hotelInfo],
    );

    const offersInfoNode = useMemo(
        () => (
            <HotelPageOffersInfo
                className={cx('gapTop')}
                hotelInfo={hotelInfo}
                isLoadingOffers={isLoadingOffers}
                onOffersButtonClick={onOffersButtonClick}
                onOfferWatchButtonClick={onOfferWatchButtonClick}
                withRoomsMatching={withRoomsMatching}
                backButtonLink={backButtonLink}
            />
        ),
        [
            hotelInfo,
            isLoadingOffers,
            onOffersButtonClick,
            backButtonLink,
            withRoomsMatching,
            onOfferWatchButtonClick,
        ],
    );

    const layout = useMemo(() => {
        if (hotelsFeaturesDown) {
            return (
                <>
                    {offersInfoNode}
                    {aboutHotelNode}
                </>
            );
        }

        return (
            <>
                {aboutHotelNode}
                <Separator margin={5} />
                {offersInfoNode}
            </>
        );
    }, [aboutHotelNode, offersInfoNode, hotelsFeaturesDown]);

    return (
        <>
            <HotelPageCardGallery
                hotelInfo={hotelInfo}
                hotelImages={hotelImages}
                getHotelImages={getHotelImages}
                className={cx('gallery')}
            />

            <TwoColumnLayout
                deviceType={deviceType}
                rightColumnOffset={RIGHT_COLUMN_OFFSET}
                rightColumnWidth={RIGHT_COLUMN_WIDTH}
            >
                <TwoColumnLayout.LeftColumn>
                    {layout}
                    {hasReviews && (
                        <HotelReviews
                            reviewsRef={reviewsRef}
                            className={cx('gapTop')}
                            deviceType={deviceType}
                        />
                    )}
                </TwoColumnLayout.LeftColumn>
                <TwoColumnLayout.RightColumn>
                    <HotelPageGeoInfo
                        hotel={hotelInfo.hotel}
                        onMapClick={onMapClick}
                    />

                    <HotelPageSimilarHotels
                        hotelInfo={hotelInfo}
                        className={cx('similarHotels')}
                    />
                    <HotelFeatures className={cx('features')} />
                </TwoColumnLayout.RightColumn>
            </TwoColumnLayout>
        </>
    );
};

export default HotelPageCardMainTabDesktop;
