import React, {FC, useCallback, useMemo, useRef} from 'react';

import {IHotelInfo} from 'server/api/HotelSearchAPI/types/IHotelInfo';
import {IGeoRegion} from 'types/hotels/hotel/IGeoRegion';

import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {useExperiments} from 'utilities/hooks/useExperiments';

import * as i18nBlock from 'i18n/hotels-HotelPage';

import HotelPageSummaryMobile from 'projects/hotels/pages/HotelPage/components/HotelPageSummaryMobile/HotelPageSummaryMobile';
import HotelPageSimilarHotels from 'projects/hotels/pages/HotelPage/components/HotelPageCardMainTab/blocks/HotelPageSimilarHotels/HotelPageSimilarHotels';
import HotelReviews from 'projects/hotels/components/HotelReviews/HotelReviews';
import HotelPageOffersInfo from 'projects/hotels/pages/HotelPage/components/HotelPageCardMainTab/blocks/HotelPageOffersInfo/HotelPageOffersInfo';
import {IHotelPageCardGalleryProps} from 'projects/hotels/pages/HotelPage/components/HotelPageCardMainTab/blocks/HotelPageCardGallery/HotelPageCardGallery';
import HotelFeaturesMobile from 'projects/hotels/pages/HotelPage/components/HotelPageCardMainTab/blocks/HotelFeaturesMobile/HotelFeaturesMobile';
import Card from 'components/Card/Card';
import HotelPageAboutHotelInfo from 'projects/hotels/pages/HotelPage/components/HotelPageCardMainTab/blocks/HotelPageAboutHotelInfo/HotelPageAboutHotelInfo';
import Heading from 'components/Heading/Heading';

import cx from './HotelPageCardMainTabMobile.scss';

interface IHotelPageCardMainTabMobileProps extends IWithQaAttributes {
    hotelInfo: IHotelInfo;

    onAddressClick: () => void;
    onReviewsButtonClick: () => void;
    onFavoritesClick: () => void;
    onOffersButtonClick: () => void;
    onOfferWatchButtonClick: (roomId: string) => void;

    hotelImages: IHotelPageCardGalleryProps['hotelImages'];
    getHotelImages: IHotelPageCardGalleryProps['getHotelImages'];

    reviewsRef: React.RefObject<HTMLDivElement>;

    withRoomsMatching?: boolean;
    isLoadingOffers: boolean;
    backButtonLink?: string;
    region?: IGeoRegion;
}

const HotelPageCardMainTabMobile: FC<IHotelPageCardMainTabMobileProps> = ({
    hotelInfo,
    onAddressClick,
    onFavoritesClick,
    hotelImages,
    getHotelImages,
    onReviewsButtonClick,
    reviewsRef,
    isLoadingOffers,
    withRoomsMatching,
    backButtonLink,
    onOffersButtonClick,
    onOfferWatchButtonClick,
    region,
    ...qaProps
}) => {
    const deviceType = useDeviceType();
    const {hotelsFeaturesDown} = useExperiments();
    const hasReviews = Boolean(hotelInfo.reviewsInfo);

    const roomsRef = useRef<HTMLDivElement>(null);

    const scrollToOffersInfo = useCallback(() => {
        roomsRef.current?.scrollIntoView();
    }, [roomsRef]);

    const offersInfoNode = useMemo(
        () => (
            <HotelPageOffersInfo
                hotelInfo={hotelInfo}
                roomsRef={roomsRef}
                isLoadingOffers={isLoadingOffers}
                onOffersButtonClick={onOffersButtonClick}
                onOfferWatchButtonClick={onOfferWatchButtonClick}
                withRoomsMatching={withRoomsMatching}
                backButtonLink={backButtonLink}
                className={cx('offers')}
                region={region}
            />
        ),
        [
            hotelInfo,
            isLoadingOffers,
            onOffersButtonClick,
            onOfferWatchButtonClick,
            withRoomsMatching,
            backButtonLink,
            region,
        ],
    );

    /*TODO: Вынести в отдельный компонент, если эксп – ок */
    const aboutHotelNode = useMemo(
        () => (
            <Card className={cx('about')}>
                <Heading level={2} className={cx('aboutTitle')}>
                    {i18nBlock.aboutHotelLabel()}
                </Heading>
                <HotelPageAboutHotelInfo
                    hotelInfo={hotelInfo}
                    {...prepareQaAttributes('aboutHotel')}
                />
            </Card>
        ),
        [hotelInfo],
    );

    const layout = useMemo(() => {
        if (!hotelsFeaturesDown) {
            return offersInfoNode;
        }

        return (
            <>
                {offersInfoNode}
                {aboutHotelNode}
            </>
        );
    }, [offersInfoNode, aboutHotelNode, hotelsFeaturesDown]);

    return (
        <>
            <HotelPageSummaryMobile
                hotelInfo={hotelInfo}
                onFavoritesClick={onFavoritesClick}
                onReviewsClick={onReviewsButtonClick}
                onAddressClick={onAddressClick}
                onOffersButtonClick={scrollToOffersInfo}
                hotelImages={hotelImages}
                getHotelImages={getHotelImages}
                className={cx('summary')}
                {...prepareQaAttributes(qaProps)}
            />

            {layout}

            <HotelFeaturesMobile className={cx('features')} />

            {hasReviews && (
                <HotelReviews
                    reviewsRef={reviewsRef}
                    className={cx('reviews')}
                    deviceType={deviceType}
                />
            )}

            <HotelPageSimilarHotels
                hotelInfo={hotelInfo}
                className={cx('similar')}
            />
        </>
    );
};

export default HotelPageCardMainTabMobile;
