import {FunctionComponent, useCallback, useMemo} from 'react';

import {HotelImagesType} from 'types/hotels/hotel/IHotelImages';
import {EHotelImageAltayStandardSize} from 'types/hotels/common/EHotelImageAltayStandardSize';
import {IWithClassName} from 'types/withClassName';

import {THotelImagesSelectorType} from 'selectors/hotels/hotel/imagesTab/getHotelImages';

/* Utils */
import {convertHotelOrRoomImages} from 'projects/hotels/utilities/prepareAndParseImages/prepareAndParseImages';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

/* Components */

import ImagesCarousel from 'components/ImagesCarousel/ImagesCarousel';

import cx from './HotelPageMobileMainImage.scss';

const MOBILE_MAIN_IMAGE_SIZE = EHotelImageAltayStandardSize.L;

export interface IHotelPageMobileMainImageProps
    extends IWithClassName,
        IWithQaAttributes {
    hotelImages: THotelImagesSelectorType;
    imagesAlt?: string;
    images: HotelImagesType;
    onImagesClick: (index: number) => void;
    loadMoreImages: () => void;
}

const HotelPageMobileMainImage: FunctionComponent<IHotelPageMobileMainImageProps> =
    ({
        images,
        imagesAlt,
        onImagesClick,
        loadMoreImages,
        hotelImages: {data},
        className,
        ...rest
    }) => {
        const convertedImages = useMemo(
            () =>
                convertHotelOrRoomImages(
                    [...images, ...data.images.slice(4)],
                    MOBILE_MAIN_IMAGE_SIZE,
                ).map(image => ({
                    ...image,
                    alt: image.alt || imagesAlt,
                })),
            [images, imagesAlt, data],
        );

        const handleImageClick = useCallback(
            (src: string, index: number) => {
                onImagesClick(index);
            },
            [onImagesClick],
        );

        const getWidth = useMemo(() => {
            if (images.length <= 1) {
                return (): string => '100%';
            }

            return undefined;
        }, [images.length]);

        return (
            <div
                className={cx('root', className)}
                {...prepareQaAttributes(rest)}
            >
                <ImagesCarousel
                    className={cx('container')}
                    imageClassName={cx('containerImage')}
                    images={convertedImages}
                    isDesktop={false}
                    onLastItemVisible={loadMoreImages}
                    onImageClick={handleImageClick}
                    getItemWidth={getWidth}
                />
            </div>
        );
    };

export default HotelPageMobileMainImage;
