import {
    FunctionComponent,
    useState,
    useEffect,
    useCallback,
    useMemo,
    useRef,
    MouseEvent,
} from 'react';

import {ARROW_LEFT, ARROW_RIGHT} from 'constants/eventKeys';
import {IOS} from 'utilities/deviceType/constants';

/* Types */
import {
    IHotelOrRoomImage,
    IHotelOrRoomImageSize,
} from 'types/hotels/hotel/IHotelImages';
import {
    ITranslationChangeData,
    IZoomChangeData,
} from 'components/Zoomable/types';
import {EHotelsGoal} from 'utilities/metrika/types/goals/hotels';

import {parseImageUrlTemplate} from 'projects/hotels/utilities/prepareAndParseImages/prepareAndParseImages';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {deviceMods} from 'utilities/stylesUtils';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {isOS} from 'utilities/deviceType/isOS';
import {useBoolean} from 'utilities/hooks/useBoolean';
import {reachGoal} from 'utilities/metrika';
import {CHAR_NBSP} from 'utilities/strings/charCodes';

import {use100vh} from 'hooks/use100vh';

/* Components */
import ModalWithHistoryBack, {
    IModalWithHistoryBackProps,
} from 'containers/withSupportHistoryBack/ModalWithHistoryBack/ModalWithHistoryBack';

import Heading from 'components/Heading/Heading';
import Carousel from 'components/Carousel/Carousel';
import ControlButton from './components/ControlButton/ControlButton';
import BottomCarousel from './components/BottomCarousel/BottomCarousel';
import ImagesViewerMainImage from './components/ImagesViewerMainImage/ImagesViewerMainImage';
import CloseIcon from 'icons/16/Close';
import HotelStars from 'components/HotelStars/HotelStars';

/* Styles */

import cx from './HotelPageImagesViewer.scss';

const MAIN_IMAGE_MOBILE_SIZE_NAME = 'XL';
const MAIN_IMAGE_SIZE_NAME = 'XXXL';

export interface IHotelPageImagesViewerProps
    extends IWithQaAttributes,
        Pick<IModalWithHistoryBackProps, 'isVisible'> {
    images: IHotelOrRoomImage[];
    totalImageCount: number;
    name?: string;
    stars?: number;
    initialIndex?: number;
    onMoreImages?: () => void;
    onCloseClick: () => void;
    imageSrcGetter?: (
        urlTemplate: string,
        size: string,
        availableSizes: IHotelOrRoomImageSize[],
    ) => string;
}

const HotelPageImagesViewer: FunctionComponent<IHotelPageImagesViewerProps> =
    props => {
        const {
            images,
            totalImageCount,
            name,
            stars,
            initialIndex,
            onCloseClick,
            onMoreImages,
            imageSrcGetter = parseImageUrlTemplate,
            isVisible,
            ...modalProps
        } = props;

        const deviceType = useDeviceType();
        const {isMobile, isDesktop} = deviceType;

        const isIOS = isOS(deviceType.os, IOS);
        const mainImageSize = isMobile
            ? MAIN_IMAGE_MOBILE_SIZE_NAME
            : MAIN_IMAGE_SIZE_NAME;

        /* State */

        const [imageIndex, setImageIndex] = useState(initialIndex || 0);

        const [swipeDisabled, setSwipeDisabled] = useState({
            disableLeftSwipe: true,
            disableRightSwipe: true,
        });
        const carouselInstance = useRef<Carousel>(null);
        const withDrag = useRef<boolean>(false);
        const {value: isZoomed, setValue: setIsZoomed} = useBoolean(false);
        const viewportHeight = use100vh();
        const {value: withoutControls, toggle: toggleControls} =
            useBoolean(false);

        const imageUrls = useMemo(
            () =>
                images &&
                images.map(i =>
                    imageSrcGetter(i.urlTemplate, mainImageSize, i.sizes),
                ),
            [images, mainImageSize, imageSrcGetter],
        );

        const mainImageUrl = useMemo(
            () => imageUrls[imageIndex],
            [imageUrls, imageIndex],
        );
        const viewerHeight =
            !isIOS && isMobile && viewportHeight ? viewportHeight : undefined;

        /* Helper */

        const handleZoomChange = useCallback(
            ({scale, minScale, maxScale}: IZoomChangeData) => {
                const threshold = (maxScale - minScale) / 10;

                // сейчас используется только для десктопа
                // toFixed - чтобы отбрасывать дробную часть, которая появляется при смене зума
                const isZoomBecameEnabled =
                    scale.toFixed(2) !== minScale.toFixed(2);

                setIsZoomed(isZoomBecameEnabled);

                if (isDesktop && isZoomBecameEnabled && !isZoomed) {
                    // считаем цель когда пользователь включил зум
                    reachGoal(
                        EHotelsGoal.HOTEL_PAGE_IMAGES_TAB_DESKTOP_ZOOM_IN,
                    );
                }

                if (scale - minScale > threshold) {
                    withDrag.current = true;
                } else {
                    withDrag.current = false;
                }
            },
            [isDesktop],
        );

        const handleTranslationChange = useCallback(
            (args: ITranslationChangeData) => {
                if (!isMobile) {
                    return;
                }

                const disableRightSwipe = !args.rightBoundaryReached;
                const disableLeftSwipe = !args.leftBoundaryReached;

                setSwipeDisabled({disableLeftSwipe, disableRightSwipe});
            },
            [isMobile],
        );

        const setImageActive = useCallback(
            (index: number, isSmooth?: boolean) => {
                setImageIndex(index);

                if (carouselInstance.current) {
                    carouselInstance.current.setItemCenter(
                        index,
                        isSmooth ? undefined : {inline: 'center'},
                    );
                }
            },
            [carouselInstance],
        );

        useEffect(() => {
            if (initialIndex !== undefined) {
                setImageActive(initialIndex);
            }
        }, [setImageActive, initialIndex, isVisible, carouselInstance]);

        useEffect(() => {
            if (initialIndex !== undefined) {
                setImageActive(initialIndex);
            }
        }, [setImageActive, initialIndex, isVisible, carouselInstance]);

        /* Handlers */

        const setPrevImageIndex = useCallback(() => {
            const index = Math.max(imageIndex - 1, 0);

            setImageActive(index, true);
        }, [setImageActive, imageIndex]);

        const setNextImageIndex = useCallback(() => {
            const index = Math.min(imageIndex + 1, images.length - 1);

            setImageActive(index, true);
        }, [setImageActive, imageIndex, images]);

        const handleCarouselImageClick = useCallback(
            (index: number) => {
                setImageIndex(index);

                if (carouselInstance.current && index !== imageIndex) {
                    const itemWidth = carouselInstance.current.getItemWidth(
                        index,
                        true,
                    );

                    if (itemWidth) {
                        carouselInstance.current.scrollBy(
                            index > imageIndex ? itemWidth : -itemWidth,
                        );
                    }
                }
            },
            [setImageIndex, carouselInstance, imageIndex],
        );

        const handleCloseClick = useCallback(
            (e: MouseEvent<HTMLSpanElement>) => {
                e.stopPropagation();
                onCloseClick?.();
            },
            [onCloseClick],
        );

        const handleKeyDown = useCallback(
            (e: KeyboardEvent) => {
                if (e.key === ARROW_LEFT) {
                    setPrevImageIndex();
                } else if (e.key === ARROW_RIGHT) {
                    setNextImageIndex();
                }
            },
            [setNextImageIndex, setPrevImageIndex],
        );

        useEffect(() => {
            if (isMobile) {
                return undefined;
            }

            document.addEventListener('keyup', handleKeyDown);

            return (): void => {
                document.removeEventListener('keyup', handleKeyDown);
            };
        }, [isMobile, handleKeyDown]);

        /* Render */

        return (
            <ModalWithHistoryBack
                fullScreen={isMobile}
                hasCloseButton={false}
                isMobile={isMobile}
                onClose={onCloseClick}
                className={cx('modal')}
                containerClassName={cx('modalContent')}
                isVisible={isVisible}
                {...modalProps}
            >
                <div
                    className={cx('root', deviceMods('root', deviceType), {
                        root_mobile_ios: isIOS,
                        root_withoutControls: withoutControls,
                    })}
                    style={{height: viewerHeight}}
                >
                    <div className={cx('mainContent')}>
                        {isVisible && ( // Zoomable не правильно определяет свои размеры если родилеь display:none
                            <ImagesViewerMainImage
                                className={cx('mainImage')}
                                deviceType={deviceType}
                                currentImageSrc={mainImageUrl}
                                currentImageIndex={imageIndex}
                                imageUrls={imageUrls}
                                onClick={isMobile ? toggleControls : undefined}
                                onSideSwipe={setImageActive}
                                onDownSwipe={onCloseClick}
                                onZoomChange={handleZoomChange}
                                onTranslationChange={handleTranslationChange}
                                isImageZoomed={isZoomed}
                                disableSwipe={swipeDisabled}
                                withDrag={withDrag}
                                {...prepareQaAttributes({
                                    parent: props,
                                    current: 'mainImage',
                                })}
                            >
                                {isDesktop && imageIndex !== 0 && (
                                    <>
                                        {/*Элементы на одном уровне, чтобы подложка была под картинкой, а кнопка над*/}
                                        <div
                                            onClick={setPrevImageIndex}
                                            className={cx(
                                                'clickable-area__button_prev',
                                            )}
                                        ></div>
                                        <ControlButton
                                            className={cx(`button_prev`)}
                                            type="prev"
                                            onClick={setPrevImageIndex}
                                            {...prepareQaAttributes({
                                                parent: props,
                                                current: 'controlButtonPrev',
                                            })}
                                        />
                                    </>
                                )}
                                {isDesktop && imageIndex < totalImageCount - 1 && (
                                    <>
                                        <div
                                            onClick={setNextImageIndex}
                                            className={cx(
                                                'clickable-area__button_next',
                                            )}
                                        ></div>
                                        <ControlButton
                                            onClick={setNextImageIndex}
                                            className={cx(`button_next`)}
                                            type="next"
                                            {...prepareQaAttributes({
                                                parent: props,
                                                current: 'controlButtonNext',
                                            })}
                                        />
                                    </>
                                )}
                            </ImagesViewerMainImage>
                        )}
                    </div>
                    {name && (
                        <Heading
                            level={3}
                            className={cx('name')}
                            {...prepareQaAttributes({
                                parent: props,
                                current: 'heading',
                            })}
                        >
                            {name}
                            {stars && (
                                <>
                                    {CHAR_NBSP}
                                    <HotelStars
                                        stars={stars}
                                        size={isMobile ? '8' : '12'}
                                    />
                                </>
                            )}
                        </Heading>
                    )}
                    <span
                        className={cx('closeButton')}
                        onClick={handleCloseClick}
                        {...prepareQaAttributes({
                            parent: props,
                            current: 'closeButton',
                        })}
                    >
                        <CloseIcon fill="#fff" />
                    </span>
                    {isDesktop && (
                        <BottomCarousel
                            className={cx('carousel')}
                            deviceType={deviceType}
                            images={images}
                            selectedIndex={imageIndex}
                            carouselRef={carouselInstance}
                            totalImageCount={totalImageCount}
                            imageSrcGetter={imageSrcGetter}
                            onLastImageVisible={onMoreImages}
                            onImageClick={handleCarouselImageClick}
                            {...prepareQaAttributes(props)}
                        />
                    )}
                    {isMobile && (
                        <span className={cx('imageIndex')}>
                            {imageIndex + 1}/{imageUrls.length}
                        </span>
                    )}
                </div>
            </ModalWithHistoryBack>
        );
    };

export default HotelPageImagesViewer;
