import {
    FunctionComponent,
    MutableRefObject,
    ReactNode,
    useCallback,
} from 'react';

/* Types */
import {IWithClassName} from 'types/withClassName';
import {IWithDeviceType} from 'types/withDeviceType';
import {UIEvent} from 'types/events';
import {
    ITranslationChangeData,
    IZoomChangeData,
} from 'components/Zoomable/types';

/* QA */
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

/* Components */
import Swipeable, {
    TDisableSwipePropsType,
} from 'components/Swipeable/Swipeable';
import ZoomableImage from 'components/ZoomableImage/ZoomableImage';

import cx from './ImagesViewerMainImage.scss';

export interface IImagesViewerMainImageProps
    extends IWithClassName,
        IWithDeviceType,
        IWithQaAttributes {
    currentImageSrc: string;
    currentImageIndex: number;
    isImageZoomed: boolean;
    imageUrls: string[];
    disableSwipe: TDisableSwipePropsType;
    withDrag?: MutableRefObject<boolean>;

    onClick?: (e: UIEvent, index?: number | undefined) => void;
    onSideSwipe: (index: number) => void;
    onDownSwipe: () => void;
    onZoomChange: (data: IZoomChangeData) => void;
    onTranslationChange?: (data: ITranslationChangeData) => void;
}

const ImagesViewerMainImage: FunctionComponent<IImagesViewerMainImageProps> = ({
    className,
    deviceType: {isMobile},
    currentImageSrc,
    currentImageIndex,
    imageUrls,
    disableSwipe,
    withDrag,
    onClick,
    onSideSwipe,
    isImageZoomed,
    onDownSwipe,
    onZoomChange,
    onTranslationChange,
    children,
    ...rest
}) => {
    const handleImageClick = useCallback(
        (e: UIEvent) => {
            onClick?.(e, currentImageIndex);
        },
        [onClick, currentImageIndex],
    );

    if (isMobile && imageUrls) {
        return (
            <Swipeable<string>
                className={className}
                sideThreshold={0.2}
                itemsData={imageUrls}
                index={currentImageIndex}
                disableSwipe={disableSwipe}
                onSideSwipe={onSideSwipe}
                onDownSwipe={onDownSwipe}
                withSwipeDown
                renderItem={({data: src, isCurrent}): ReactNode => (
                    <ZoomableImage
                        src={src}
                        imageClassName={cx('image')}
                        withoutImageClassName={cx('withoutImage')}
                        onClick={handleImageClick}
                        onZoomChange={onZoomChange}
                        disableZoom={!isCurrent}
                        withDrag={withDrag}
                        onTranslationChange={onTranslationChange}
                        {...prepareQaAttributes(rest)}
                    />
                )}
            />
        );
    }

    return (
        <ZoomableImage
            className={className}
            src={currentImageSrc}
            imageClassName={cx('image', {
                'image--zoomed': isImageZoomed,
            })}
            withoutImageClassName={cx('withoutImage')}
            onClick={handleImageClick}
            onZoomChange={onZoomChange}
            onTranslationChange={onTranslationChange}
            withDrag={withDrag}
            {...prepareQaAttributes(rest)}
        >
            {children}
        </ZoomableImage>
    );
};

export default ImagesViewerMainImage;
