import React, {FunctionComponent, useMemo} from 'react';
import {IYMapsApi} from 'react-yandex-maps';
import {renderToString} from 'react-dom/server';

import {IWithClassName} from 'types/withClassName';
import {IHotel} from 'types/hotels/hotel/IHotel';

import {IWithQaAttributes} from 'utilities/qaAttributes/qaAttributes';
import {useExperiments} from 'utilities/hooks/useExperiments';

/* Components */
import {YandexMapPlacemark} from 'components/YandexMaps/YandexMaps';
import HotelPageMapMarkerContent from 'projects/hotels/pages/HotelPage/components/HotelPageMapMarker/HotelPageMapMarkerContent/HotelPageMapMarkerContent';

export interface IHotelPageMapMarkerProps
    extends IWithClassName,
        IWithQaAttributes {
    hotel: IHotel;
    yandexMapAPIInstance: IYMapsApi | null;
    canRenderAddress: boolean;
}

const HotelPageMapMarker: FunctionComponent<IHotelPageMapMarkerProps> =
    props => {
        const {hotel, yandexMapAPIInstance, canRenderAddress} = props;
        const {coordinates} = hotel;
        const {hotelsPercentDiscount} = useExperiments();

        const markerContentLayout = useMemo(() => {
            function renderMarker(): React.ReactElement {
                return (
                    <HotelPageMapMarkerContent
                        hotelsPercentDiscount={hotelsPercentDiscount}
                        {...props}
                    />
                );
            }

            if (yandexMapAPIInstance) {
                return yandexMapAPIInstance.templateLayoutFactory.createClass(
                    renderToString(renderMarker()),
                    {
                        build: function () {
                            this.constructor.superclass.build.call(this);
                            this._markerNode = this.getElement().firstChild;
                            this._parentElementNode = this.getParentElement();
                        },
                        getShape: function () {
                            const {
                                offsetWidth: markerWidth,
                                offsetHeight: markerHeight,
                            } = this._markerNode;

                            return new yandexMapAPIInstance.shape.Rectangle(
                                new yandexMapAPIInstance.geometry.pixel.Rectangle(
                                    [
                                        [markerWidth / 2, 0],
                                        [-markerHeight / 2, 0],
                                    ],
                                ),
                            );
                        },
                    },
                );
            }
        }, [hotel, yandexMapAPIInstance, canRenderAddress]);

        return (
            <YandexMapPlacemark
                geometry={[coordinates.lat, coordinates.lon]}
                options={{
                    iconLayout: markerContentLayout,
                }}
            />
        );
    };

export default HotelPageMapMarker;
