import React, {PureComponent, ReactNode} from 'react';

import {IWithClassName} from 'types/withClassName';
import {IWithDeviceType} from 'types/withDeviceType';
import {IHotelOrRoomImage} from 'types/hotels/hotel/IHotelImages';
import {EHotelImageAltayStandardSize} from 'types/hotels/common/EHotelImageAltayStandardSize';

import {
    convertHotelOrRoomImages,
    convertUrlTemplateImagesToFlatList,
} from 'projects/hotels/utilities/prepareAndParseImages/prepareAndParseImages';
import {deviceMods} from 'utilities/stylesUtils';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import ImagesCarousel from 'components/ImagesCarousel/ImagesCarousel';
import Text from 'components/Text/Text';

import GalleryRow from '../GalleryRow/GalleryRow';

import cx from './HotelPageRoomGallery.scss';

/* Constants */
const START_TILE_IMAGES_INDEX = 1;
const TOTAL_TILE_IMAGES = 5;
const TOTAL_GALLERY_IMAGES = START_TILE_IMAGES_INDEX + TOTAL_TILE_IMAGES;
const DESKTOP_IMAGES_SPACE_BETWEEN = 0;
const MOBILE_IMAGES_SPACE_BETWEEN = 1;

/* Types */
interface IHotelPageRoomGalleryProps
    extends IWithClassName,
        IWithDeviceType,
        IWithQaAttributes {
    images: IHotelOrRoomImage[];
    onImageClick?: (imageIndex: number) => void;
    imagesAlt?: string;
}

class HotelPageRoomGallery extends PureComponent<IHotelPageRoomGalleryProps> {
    static defaultProps = {
        images: [],
    };

    private handleCarouselImageClick = (src: string, index: number): void => {
        const {onImageClick} = this.props;

        if (onImageClick) {
            onImageClick(index);
        }
    };

    private handleRowImageClick = (index: number): void => {
        const {onImageClick} = this.props;

        if (onImageClick) {
            onImageClick(index + START_TILE_IMAGES_INDEX);
        }
    };

    private renderImagesCarousel(): ReactNode {
        const {images, deviceType} = this.props;
        const {isDesktop} = deviceType;
        const convertedImages = convertHotelOrRoomImages(
            images,
            EHotelImageAltayStandardSize.L,
        );

        return (
            <ImagesCarousel
                className={cx(
                    'imagesCarousel',
                    deviceMods('imagesCarousel', deviceType),
                )}
                imageClassName={cx('carouselImage')}
                type={isDesktop ? 'mini' : 'full'}
                arrowsType="dark"
                spaceBetween={
                    isDesktop
                        ? DESKTOP_IMAGES_SPACE_BETWEEN
                        : MOBILE_IMAGES_SPACE_BETWEEN
                }
                images={convertedImages}
                onImageClick={this.handleCarouselImageClick}
                isDesktop={isDesktop}
                autoHeight={isDesktop}
                lazyLoad
            />
        );
    }

    private renderTotalCount = (totalImageCount: number): ReactNode => {
        const otherImageCount =
            totalImageCount - TOTAL_TILE_IMAGES - START_TILE_IMAGES_INDEX;

        return (
            <Text size="m" color="inverse">
                +{otherImageCount}
            </Text>
        );
    };

    private renderTileImages(): ReactNode {
        const {images, imagesAlt} = this.props;
        const totalImageCount = images.length;
        const sizedImagesUrls = convertUrlTemplateImagesToFlatList(
            images,
            EHotelImageAltayStandardSize.S,
        ).slice(START_TILE_IMAGES_INDEX, TOTAL_GALLERY_IMAGES);

        return (
            <GalleryRow
                className={cx('tileImages')}
                imageWrapperClassName={cx('tileImage')}
                imagesAlt={imagesAlt}
                isLastClickable={totalImageCount > TOTAL_GALLERY_IMAGES}
                imageSources={sizedImagesUrls}
                totalImageCount={totalImageCount}
                renderTotalCount={this.renderTotalCount}
                onImageClick={this.handleRowImageClick}
                onMoreImagesClick={this.handleRowImageClick}
            />
        );
    }

    render(): React.ReactNode {
        const {
            images,
            className,
            deviceType: {isDesktop},
        } = this.props;

        if (images.length > 0) {
            return (
                <div
                    className={cx(className, 'roomGallery')}
                    {...prepareQaAttributes(this.props)}
                >
                    {this.renderImagesCarousel()}
                    {isDesktop && this.renderTileImages()}
                </div>
            );
        }

        return null;
    }
}

export default HotelPageRoomGallery;
