import {FC, useMemo} from 'react';
import {useSelector} from 'react-redux';

import {IWithDeviceType} from 'types/withDeviceType';
import {
    ESearchFormSize,
    ESearchFormTriggerViewType,
} from 'components/SearchForm/types';
import EAsyncStatus from 'types/common/EAsyncStatus';

import {getHotelCalendarPrices} from 'selectors/hotels/hotel/getHotelCalendarPrices';

import {deviceModMobile} from 'utilities/stylesUtils';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import * as i18nBlock from 'i18n/hotels-HotelPageSearchForm';

import SearchForm from 'projects/hotels/components/SearchForm/SearchForm';
import {TButtonTheme} from 'components/Button/Button';
import {IFormFieldsRef} from 'components/SearchForm/SearchForm';
import Spinner from 'components/Spinner/Spinner';
import Flex from 'components/Flex/Flex';

import cx from './HotelPageSearchForm.scss';

/* Component Types */

export interface IHotelPageSearchFormProps
    extends IWithDeviceType,
        IWithQaAttributes {
    size?: ESearchFormSize;
    submitButtonTheme?: TButtonTheme;
    submitButtonText?: string;
    formRef?: React.RefObject<HTMLFormElement>;
    fieldsRef?: React.MutableRefObject<IFormFieldsRef | undefined>;
}

const HotelPageSearchForm: FC<IHotelPageSearchFormProps> = props => {
    const {
        deviceType,
        size,
        submitButtonTheme,
        submitButtonText,
        formRef,
        fieldsRef,
    } = props;
    const {isDesktop} = deviceType;
    const calendarPrices = useSelector(getHotelCalendarPrices);

    const messageNode = useMemo(() => {
        if (calendarPrices.status === EAsyncStatus.LOADING) {
            return (
                <Flex between={2} inline>
                    <Spinner size="xxs" />
                    <span>{i18nBlock.loadingPrices()}</span>
                </Flex>
            );
        }

        if (calendarPrices.status === EAsyncStatus.SUCCESS) {
            return i18nBlock.approximatePricesForOneNight();
        }

        return null;
    }, [calendarPrices]);

    const datePickerFooterBlockParams = useMemo(
        () => ({
            messageNode,
            visible: Boolean(messageNode),
        }),
        [messageNode],
    );

    if (deviceType.isIe) {
        return null;
    }

    return (
        <SearchForm
            searchFormClassName={cx(deviceModMobile('searchForm', deviceType))}
            travellersClassName={cx(deviceModMobile('travellers', deviceType))}
            travellersTriggerClassName={cx(
                deviceModMobile('travellersTrigger', deviceType),
            )}
            datePickerTriggerClassName={cx(
                deviceModMobile('datePickerTrigger', deviceType),
            )}
            submitButtonClassName={cx(
                'submitButton',
                deviceModMobile('submitButton', deviceType),
            )}
            size={size}
            triggerViewType={ESearchFormTriggerViewType.TILE}
            submitButtonTheme={submitButtonTheme || 'secondary'}
            hasDelayBeforeUpdateLocation
            hasDirectionSuggest={false}
            hideDateClearButton
            submitButtonText={submitButtonText || i18nBlock.submitButton()}
            canSubmitOnChange={isDesktop}
            canToggleDropdowns={isDesktop}
            isFullDateFormat={isDesktop}
            formRef={formRef}
            fieldsRef={fieldsRef}
            calendarPrices={calendarPrices.data}
            datePickerFooterBlockParams={datePickerFooterBlockParams}
            {...prepareQaAttributes(props)}
        />
    );
};

export default HotelPageSearchForm;
