import {FC} from 'react';

import {IHotelInfo} from 'server/api/HotelSearchAPI/types/IHotelInfo';
import {IWithClassName} from 'types/withClassName';

import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import checkOffersDiscount from 'projects/hotels/pages/HotelPage/utilities/checkOffersDiscount';
import {useExperiments} from 'utilities/hooks/useExperiments';

import HotelTitle from 'projects/hotels/pages/HotelPage/components/HotelTitle/HotelTitle';
import HotelRatingWithReviewLink from 'projects/hotels/components/HotelRatingWithReviewLink/HotelRatingWithReviewLink';
import Separator from 'components/Separator/Separator';
import Flex from 'components/Flex/Flex';
import Card from 'components/Card/Card';
import HotelAddress from 'projects/hotels/components/HotelAddress/HotelAddress';
import Location from 'icons/16/Location';
import HotelGeoFeature from 'projects/hotels/components/HotelGeoFeature/HotelGeoFeature';
import TransportAccessibility from 'projects/hotels/components/TransportAccessibility/TransportAccessibility';
import HotelPageAboutHotelInfo from 'projects/hotels/pages/HotelPage/components/HotelPageCardMainTab/blocks/HotelPageAboutHotelInfo/HotelPageAboutHotelInfo';
import HotelPageSummaryOffersButton from './components/HotelPageSummaryOffersButton/HotelPageSummaryOffersButton';

import HotelPageCardGallery, {
    IHotelPageCardGalleryProps,
} from '../HotelPageCardMainTab/blocks/HotelPageCardGallery/HotelPageCardGallery';

import cx from './HotelPageSummaryMobile.scss';

interface IHotelPageSummaryMobileProps
    extends IWithClassName,
        IWithQaAttributes {
    hotelInfo: IHotelInfo;
    onFavoritesClick: () => void;
    onReviewsClick: () => void;
    onOffersButtonClick: () => void;
    onAddressClick: () => void;
    hotelImages: IHotelPageCardGalleryProps['hotelImages'];
    getHotelImages: IHotelPageCardGalleryProps['getHotelImages'];
}

const HotelPageSummaryMobile: FC<IHotelPageSummaryMobileProps> = ({
    onReviewsClick,
    hotelInfo,
    onOffersButtonClick,
    hotelImages,
    getHotelImages,
    onAddressClick,
    onFavoritesClick,
    className,
    ...qaProps
}) => {
    const deviceType = useDeviceType();

    const {hotelsFeaturesDown} = useExperiments();

    const {
        hotel,
        offersInfo: {aggregatedOfferInfo, partnerOffers, mainOffers},
    } = hotelInfo;

    const {
        isFavorite,
        name,
        stars,
        rating,
        totalTextReviewCount,
        address,
        geoFeature,
        nearestStations,
    } = hotel;

    return (
        <Card className={cx('root', className)}>
            <div className={cx('wrapper')}>
                <HotelTitle
                    name={name}
                    stars={stars}
                    isFavorite={isFavorite}
                    handleFavoriteClick={onFavoritesClick}
                    deviceType={deviceType}
                />

                <HotelRatingWithReviewLink
                    className={cx('rating')}
                    rating={rating}
                    totalTextReviewCount={totalTextReviewCount}
                    onTotalReviewLinkClick={onReviewsClick}
                    isMobile
                />
            </div>

            <HotelPageCardGallery
                className={cx('gallery')}
                hotelInfo={hotelInfo}
                hotelImages={hotelImages}
                getHotelImages={getHotelImages}
            />

            <div className={cx('wrapper')}>
                <HotelPageSummaryOffersButton
                    className={cx('viewOffers')}
                    onClick={onOffersButtonClick}
                    minPrice={aggregatedOfferInfo?.minPrice}
                    hasDiscount={checkOffersDiscount(mainOffers, partnerOffers)}
                />
                <Separator className={cx('separator')} />
                <HotelAddress
                    className={cx('address', {address_mobile: true})}
                    text={address}
                    type="separate-link"
                    onClick={onAddressClick}
                    icon={Location}
                    textClassName={cx('addressText')}
                    multiLine
                />

                <Flex
                    className={cx('featureAndTransportAccessibility')}
                    between={4}
                    inline
                >
                    <HotelGeoFeature
                        geoFeature={geoFeature}
                        {...prepareQaAttributes({
                            current: 'geoFeature',
                            parent: qaProps,
                        })}
                    />
                    {nearestStations?.[0] && (
                        <TransportAccessibility
                            bigIcon
                            station={nearestStations[0]}
                            {...prepareQaAttributes({
                                current: 'transportAccessibility',
                                parent: qaProps,
                            })}
                        />
                    )}
                </Flex>

                {!hotelsFeaturesDown && (
                    <HotelPageAboutHotelInfo
                        className={cx('aboutHotel')}
                        hotelInfo={hotelInfo}
                        {...prepareQaAttributes('aboutHotel')}
                    />
                )}
            </div>
        </Card>
    );
};

export default HotelPageSummaryMobile;
