import React, {
    Component,
    createRef,
    MutableRefObject,
    ReactNode,
    RefObject,
} from 'react';

import {
    BOY_META_MAIN_TAB_MAX_LIMITS,
    BOY_META_MAIN_TAB_MAX_OFFERS,
    DEFAULT_ADULTS_COUNT,
} from 'constants/hotels';
import {HOTEL_OFFERS} from 'constants/urls/anchors';

import {IWithDeviceType} from 'types/withDeviceType';
import {
    IHotelOffer,
    IRequiredOfferParams,
} from 'types/hotels/offer/IHotelOffer';
import {
    FilterType,
    HotelOffersFiltersType,
    IHotelOffersInfo,
} from 'reducers/hotels/hotelPage/hotelInfo/types';
import {
    ESearchFormFieldName,
    ESearchFormSize,
} from 'components/SearchForm/types';
import {IWithClassName} from 'types/withClassName';
import {IGeoRegion} from 'types/hotels/hotel/IGeoRegion';

import {deviceMods} from 'utilities/stylesUtils';
import humanizePeriod from 'utilities/dateUtils/humanizePeriod';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {getTotalNights} from 'projects/hotels/utilities/calculateTotalNights/calculateTotalNights';
import getGuestsCount from 'projects/hotels/pages/HotelPage/utilities/getGuestsCount';
import scrollToNode from 'utilities/dom/scrollToNode';

import * as i18nBlock from 'i18n/hotels-HotelPageOffersInfo';

import Heading from 'components/Heading/Heading';
import Card from 'components/Card/Card';
import ButtonLink from 'components/ButtonLink/ButtonLink';
import Text from 'components/Text/Text';
import MainOffers from './components/MainOffers/MainOffers';
import PartnerOffers from './components/PartnerOffers/PartnerOffers';
import MainOffersSkeleton from './components/MainOffersSkeleton/MainOffersSkeleton';
import PartnerOffersSkeleton from './components/PartnerOffersSkeleton/PartnerOffersSkeleton';
import HotelPageSearchForm from 'projects/hotels/pages/HotelPage/components/HotelPageSearchForm/HotelPageSearchForm';
import Rooms from 'projects/hotels/pages/HotelPage/components/OffersInfo/components/Rooms/Rooms';
import Flex from 'components/Flex/Flex';
import {IFormFieldsRef} from 'components/SearchForm/SearchForm';
import Anchor from 'components/Anchor/Anchor';
import EmptyOffers from 'projects/hotels/pages/HotelPage/components/OffersInfo/components/EmptyOffers/EmptyOffers';

import SpecifyDatesWarning from '../SpecifyDatesWarning/SpecifyDatesWarning';

import cx from './OffersInfo.scss';

export interface IHotelOfferInfoProps
    extends IWithClassName,
        IWithDeviceType,
        IWithQaAttributes {
    canShowMainOffers?: boolean;
    canShowPartnerOffers?: boolean;

    searchParams?: IRequiredOfferParams;
    isLoading?: boolean;
    offersInfo?: IHotelOffersInfo;
    filteredOffers?: IHotelOffer[];
    isRoomGroup?: boolean;
    isFullOffersTab?: boolean;
    filters?: HotelOffersFiltersType;
    onFilterSelect?: (filter: FilterType) => void;
    onOfferSelect?: () => void;
    onOfferWatchButtonClick?: (roomId: string) => void;
    onAllOffersClick?: () => void;
    backButtonLink?: string;
    chosenId?: string;
    roomsRef?: React.RefObject<HTMLDivElement>;
    withRoomsMatching?: boolean;
    region?: IGeoRegion;
}

class OffersInfo extends Component<IHotelOfferInfoProps> {
    searchFormRef: RefObject<HTMLFormElement> = createRef();
    searchFormFieldsRef: MutableRefObject<IFormFieldsRef | undefined> = {
        current: undefined,
    };

    static defaultProps = {
        canShowMainOffers: true,
        canShowPartnerOffers: true,
    };

    private getPartnerOffersTitle(hasMainOffers: boolean): string {
        const {searchParams, deviceType} = this.props;

        if (hasMainOffers || !searchParams || deviceType.isMobile) {
            return i18nBlock.partnerTitleShort();
        }

        const {
            checkinDate,
            checkoutDate,
            adults,
            childrenAges: {length: children},
        } = searchParams;

        return i18nBlock.partnerTitleDetailed({
            dates: humanizePeriod(checkinDate, checkoutDate),
            adults,
            children,
        });
    }

    private handleCheckPricesClick = (): void => {
        if (this.searchFormRef.current) {
            scrollToNode(this.searchFormRef.current, {
                shiftY: -120,
                behavior: 'smooth',
            });
        }

        if (this.searchFormFieldsRef.current) {
            this.searchFormFieldsRef.current.focusFieldByName(
                ESearchFormFieldName.START_DATE,
            );
        }
    };

    private getTotalNights(): number | undefined {
        const {searchParams} = this.props;

        return searchParams
            ? getTotalNights(
                  searchParams.checkinDate,
                  searchParams.checkoutDate,
              )
            : undefined;
    }

    private renderAllOffersLink(): ReactNode {
        const {
            onAllOffersClick,
            offersInfo,
            deviceType: {isMobile},
        } = this.props;

        const offerCount = offersInfo?.mainOffers?.length;

        if (
            onAllOffersClick &&
            offerCount &&
            offerCount >
                BOY_META_MAIN_TAB_MAX_OFFERS + BOY_META_MAIN_TAB_MAX_LIMITS
        ) {
            return (
                <div className={cx('showMore')}>
                    <ButtonLink
                        onClick={onAllOffersClick}
                        {...prepareQaAttributes('allRoomsLink')}
                        theme="primary"
                        size="l"
                        width={isMobile ? 'max' : 'auto'}
                    >
                        {i18nBlock.allOffersWithCount({offerCount})}
                    </ButtonLink>
                </div>
            );
        }

        return null;
    }

    private renderPartnerOffers(hasMainOffers: boolean): ReactNode {
        const {offersInfo, deviceType, onOfferSelect} = this.props;
        const {isMobile} = deviceType;

        if (!offersInfo) {
            return null;
        }

        const {partnerOffers, operatorById} = offersInfo;

        if (!partnerOffers) {
            return null;
        }

        const nightsCount = this.getTotalNights();

        const offersNode = (
            <PartnerOffers
                className={cx('offers')}
                hasMainOffers={hasMainOffers}
                offers={partnerOffers}
                operatorById={operatorById}
                onOfferSelect={onOfferSelect}
                nightsCount={nightsCount}
                {...prepareQaAttributes({
                    parent: this.props,
                    current: 'partnerOffers',
                })}
            />
        );

        const Wrapper = isMobile ? Card : 'div';

        return (
            <Wrapper className={cx('partnersOffersWrapper')}>
                {this.renderPartnerTitle(hasMainOffers)}
                {offersNode}
            </Wrapper>
        );
    }

    renderOffers(): ReactNode {
        const {
            offersInfo,
            isFullOffersTab,
            deviceType,
            onOfferSelect,
            isRoomGroup = false,
            onOfferWatchButtonClick,
            chosenId,
        } = this.props;
        const {isMobile} = deviceType;

        if (offersInfo) {
            const {mainOffers, partnerOffers, operatorById} = offersInfo;
            const nightsCount = this.getTotalNights();

            const offersNode = (
                <>
                    <MainOffers
                        className={cx('offers')}
                        deviceType={deviceType}
                        offers={mainOffers || []}
                        partnerOffers={partnerOffers || []}
                        operatorById={operatorById}
                        onOfferSelect={onOfferSelect}
                        isFullOffersTab={isFullOffersTab}
                        nightsCount={nightsCount}
                        isRoomGroup={isRoomGroup}
                        onOfferWatchButtonClick={onOfferWatchButtonClick}
                        chosenId={chosenId}
                        {...prepareQaAttributes({
                            parent: this.props,
                            current: 'mainOffers',
                        })}
                    />
                </>
            );

            return isMobile ? (
                <>
                    <Card shadow="none" className={cx('mainOffersWrapper')}>
                        {offersNode}
                    </Card>
                    {!isFullOffersTab && this.renderAllOffersLink()}
                </>
            ) : (
                <>
                    {offersNode}
                    {!isFullOffersTab && this.renderAllOffersLink()}
                </>
            );
        }

        return null;
    }

    renderRooms(): ReactNode {
        const {
            offersInfo,
            onOfferSelect,
            searchParams,
            onOfferWatchButtonClick,
            chosenId,
        } = this.props;

        if (offersInfo && offersInfo.rooms) {
            const nightsCount = this.getTotalNights();
            const canRenderRoomImage = offersInfo.rooms.some(
                room => room.images && room.images.length > 0,
            );

            return (
                <Rooms
                    className={cx('roomsWrapper')}
                    offersInfo={offersInfo}
                    onOfferSelect={onOfferSelect}
                    canShowImages={canRenderRoomImage}
                    nightsCount={nightsCount}
                    onOfferWatchButtonClick={onOfferWatchButtonClick}
                    onCheckPricesClick={this.handleCheckPricesClick}
                    chosenRoomId={chosenId}
                    {...prepareQaAttributes({
                        parent: this.props,
                        current: 'rooms',
                    })}
                    searchParams={searchParams}
                />
            );
        }

        return null;
    }

    renderWithoutDatesPageTitle(): ReactNode {
        return (
            <Flex alignItems="center" between={3} inline>
                <Heading
                    level={2}
                    className={cx('titleWithoutDates')}
                    {...prepareQaAttributes('offersTitle')}
                >
                    {i18nBlock.availabilityOfRooms()}
                </Heading>
                <SpecifyDatesWarning />
            </Flex>
        );
    }

    renderMainTitle(): ReactNode {
        const {searchParams} = this.props;

        if (!searchParams) {
            return this.renderWithoutDatesPageTitle();
        }

        const {
            checkinDate,
            checkoutDate,
            adults,
            childrenAges: {length: children},
        } = searchParams;

        return (
            <Heading
                level={2}
                className={cx('title')}
                {...prepareQaAttributes({
                    parent: this.props,
                    current: 'mainOffersTitle',
                })}
            >
                {i18nBlock.heading({
                    dates: humanizePeriod(checkinDate, checkoutDate),
                    adults,
                    children,
                })}
            </Heading>
        );
    }

    renderPartnerTitle(hasMainOffers: boolean): ReactNode {
        const title = this.getPartnerOffersTitle(hasMainOffers);

        return (
            <Heading
                level={2}
                className={cx('partnerOffersTitle')}
                {...prepareQaAttributes({
                    parent: this.props,
                    current: 'partnerOffersTitle',
                })}
            >
                {title}
            </Heading>
        );
    }

    private renderFilteredOffersEmpty(): ReactNode {
        return (
            <div
                className={cx('empty')}
                {...prepareQaAttributes({
                    parent: this.props,
                    current: 'filteredOffersEmpty',
                })}
            >
                <Text size="m" weight="medium" className={cx('textEmpty')}>
                    {i18nBlock.emptyDotFiltersDotFirstLine()}
                </Text>
                <Text size="m" className={cx('textEmpty')}>
                    {i18nBlock.emptyDotFiltersDotSecondLine()}
                </Text>
            </div>
        );
    }

    private renderEmptyOffers(): ReactNode {
        const {backButtonLink, region, searchParams} = this.props;

        if (!searchParams) {
            return null;
        }

        const guestsCount = searchParams
            ? getGuestsCount(searchParams)
            : DEFAULT_ADULTS_COUNT;

        return (
            <EmptyOffers
                guestsCount={guestsCount}
                backButtonLink={backButtonLink}
                searchParams={searchParams}
                region={region}
                {...prepareQaAttributes({
                    parent: this.props,
                    current: 'emptyOffers',
                })}
            />
        );
    }

    private renderControls(): ReactNode {
        const {deviceType, isFullOffersTab, searchParams} = this.props;
        const withoutDates = !searchParams;

        return (
            <div
                className={cx('searchForm', {
                    searchForm_gray: isFullOffersTab,
                    searchForm_withoutDates: withoutDates,
                })}
                {...prepareQaAttributes('hotelPageSearchForm')}
            >
                <HotelPageSearchForm
                    deviceType={deviceType}
                    size={ESearchFormSize.M}
                    submitButtonTheme={withoutDates ? 'primary' : 'secondary'}
                    submitButtonText={
                        withoutDates ? i18nBlock.checkPrices() : undefined
                    }
                    formRef={this.searchFormRef}
                    fieldsRef={this.searchFormFieldsRef}
                    {...prepareQaAttributes({
                        parent: this.props,
                        current: 'hotelPageSearchForm',
                    })}
                />
            </div>
        );
    }

    private renderControlsWithTitle(title?: ReactNode): ReactNode {
        const controls = this.renderControls();
        const Wrapper = this.props.deviceType.isMobile ? Card : 'div';

        return (
            <Wrapper className={cx('controlsWrapper')}>
                {title}
                {controls}
            </Wrapper>
        );
    }

    private renderFlatOfferListWithFilters(): ReactNode {
        const {
            offersInfo = {mainOffers: []},
            filteredOffers = [],
            filters = {},
        } = this.props;
        const {mainOffers = []} = offersInfo;

        if (!mainOffers) {
            return null;
        }

        const hasOffersToShow = mainOffers.length > 0;
        const hasFiltersAndOffers =
            Object.keys(filters).length > 0 ? filteredOffers.length > 0 : true;

        if (hasOffersToShow) {
            return (
                <>
                    {hasFiltersAndOffers
                        ? this.renderOffers()
                        : this.renderFilteredOffersEmpty()}
                </>
            );
        }

        return null;
    }

    private renderContentByGroupType(): ReactNode {
        const {offersInfo} = this.props;

        const hasRooms = Boolean(offersInfo?.rooms?.length);
        const hasMainOffers = Boolean(offersInfo?.mainOffers?.length);

        const emptyMainOffers = hasMainOffers ? null : this.renderEmptyOffers();

        const content = hasRooms
            ? this.renderRooms()
            : this.renderFlatOfferListWithFilters();

        return (
            <>
                {emptyMainOffers}
                {content}
            </>
        );
    }

    private renderContent(): ReactNode {
        const {
            offersInfo,
            deviceType,
            isFullOffersTab,
            roomsRef,
            isLoading,
            searchParams,
            withRoomsMatching,
        } = this.props;
        const isFinished =
            offersInfo?.offerSearchProgress?.finished && !isLoading;

        const hasMainOffers = Boolean(offersInfo?.mainOffers?.length);
        const hasPartnerOffers = Boolean(offersInfo?.partnerOffers?.length);
        const hasRooms = Boolean(offersInfo?.rooms?.length);

        const renderMainOffers = isFinished;
        const renderPartnerOffers =
            isFinished && hasPartnerOffers && !(hasRooms && withRoomsMatching);
        const isWithoutDatesPage = isFinished && !searchParams && hasRooms;

        if (!isFinished) {
            return (
                <>
                    <div className={cx('root', deviceMods('root', deviceType))}>
                        {this.renderControlsWithTitle(this.renderMainTitle())}
                        <MainOffersSkeleton
                            className={cx('mainOffersSkeleton')}
                            isFullOffersTab={isFullOffersTab}
                        />
                    </div>
                    <div
                        className={cx(
                            'root',
                            'partnerOffersSkeleton',
                            deviceMods('root', deviceType),
                        )}
                    >
                        {deviceType.isDesktop &&
                            this.renderPartnerTitle(hasMainOffers)}
                        <PartnerOffersSkeleton />
                    </div>
                </>
            );
        }

        if (isWithoutDatesPage) {
            return (
                <div
                    ref={roomsRef}
                    className={cx('root', deviceMods('root', deviceType))}
                >
                    {this.renderControlsWithTitle(
                        this.renderWithoutDatesPageTitle(),
                    )}
                    {this.renderRooms()}
                </div>
            );
        }

        return (
            <>
                {renderMainOffers && (
                    <div
                        ref={roomsRef}
                        className={cx('root', deviceMods('root', deviceType))}
                        {...prepareQaAttributes('hotelMainOffers')}
                    >
                        {this.renderControlsWithTitle(this.renderMainTitle())}
                        {this.renderContentByGroupType()}
                    </div>
                )}
                {renderPartnerOffers && (
                    <div
                        ref={renderMainOffers ? null : roomsRef}
                        className={cx('root', deviceMods('root', deviceType))}
                        {...prepareQaAttributes('hotelPartnerOffers')}
                    >
                        {this.renderPartnerOffers(hasMainOffers)}
                    </div>
                )}
            </>
        );
    }

    render(): ReactNode {
        const {className} = this.props;

        const content = this.renderContent();

        if (!content) {
            return null;
        }

        return (
            <div className={className} {...prepareQaAttributes(this.props)}>
                <Anchor anchorId={HOTEL_OFFERS} />
                {content}
            </div>
        );
    }
}

export default OffersInfo;
