import {FC, MouseEventHandler, useCallback} from 'react';

import {EAppActions} from 'constants/platforms/TPlatforms';

import {IRequiredOfferParams} from 'types/hotels/offer/IHotelOffer';
import {IGeoRegion} from 'types/hotels/hotel/IGeoRegion';

import {deviceMods} from 'utilities/stylesUtils';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {useDeviceType} from 'utilities/hooks/useDeviceType';

import * as i18nBlock from 'i18n/hotels-HotelPageOffersInfo';

import Box from 'components/Box/Box';
import Text from 'components/Text/Text';
import ButtonLink from 'components/ButtonLink/ButtonLink';
import {ContainerElement} from '@yandex-lego/components/Link/Link';

import {useCoordinator} from 'contexts/PlatformContext';

import cx from './EmptyOffers.scss';

export interface IEmptyOffersProps extends IWithQaAttributes {
    region?: IGeoRegion;
    guestsCount: number;
    backButtonLink?: string;
    searchParams?: IRequiredOfferParams;
}

const EmptyOffers: FC<IEmptyOffersProps> = ({
    guestsCount,
    backButtonLink,
    searchParams,
    region,
    ...rest
}) => {
    const deviceType = useDeviceType();
    const coordinator = useCoordinator();

    const handleOnLinkClick = useCallback<MouseEventHandler<ContainerElement>>(
        event => {
            if (!searchParams) {
                return;
            }

            const processed = coordinator.doAction(
                EAppActions.NAVIGATE_TO_SEARCH_RESULT_HOTELS_PAGE,
                {
                    region,
                    searchParams,
                },
            );

            if (processed) {
                event.preventDefault();
            }
        },
        [coordinator, region, searchParams],
    );

    return (
        <div
            className={cx('root', deviceMods('root', deviceType))}
            {...prepareQaAttributes({
                parent: rest,
            })}
        >
            <Box below={4}>
                <Text size="m" weight="medium" className={cx('text')}>
                    {i18nBlock.emptyFirstLineNew({guestsCount})}
                </Text>
                <Text size="m" className={cx('text')}>
                    {i18nBlock.emptySecondLineNew()}
                </Text>
            </Box>
            <ButtonLink
                theme="secondary"
                size={deviceType.isMobile ? 'l' : 'm-inset'}
                to={backButtonLink}
                onClick={handleOnLinkClick}
            >
                {i18nBlock.emptyButtonNew()}
            </ButtonLink>
        </div>
    );
};

export default EmptyOffers;
