import {useCallback, ReactNode, FunctionComponent, useMemo} from 'react';
import {useLocation} from 'react-router-dom';
import {useSelector} from 'react-redux';
import {groupBy} from 'lodash';

import {
    BOY_META_MAIN_TAB_MAX_LIMITS,
    BOY_META_MAIN_TAB_MAX_OFFERS,
} from 'constants/hotels';

import {HotelRoomOfferType} from 'reducers/hotels/hotelPage/hotelInfo/types';
import {IWithClassName} from 'types/withClassName';
import {IWithDeviceType} from 'types/withDeviceType';
import {IHotelOffer, IHotelPartnerOffer} from 'types/hotels/offer/IHotelOffer';
import {OperatorByIdType} from 'types/hotels/offer/IHotelsOperator';

import experimentsSelector from 'selectors/common/experimentsSelector';

import {getViewModePageParamsByQuery} from 'projects/hotels/utilities/getHotelPageParams/getHotelPageParams';
import getQueryByLocation from 'utilities/getQueryByLocation/getQueryByLocation';
import {
    prepareQaAttributes,
    IWithQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {deviceMods} from 'utilities/stylesUtils';

import * as i18nBlock from 'i18n/hotels-HotelPageOffer';

/* Components */
import Text from 'components/Text/Text';
import Flex from 'components/Flex/Flex';
import YandexOffer from './components/YandexOffer/YandexOffer';
import MatchedOffers from './components/MatchedOffers/MatchedOffers';

/* Init Styles */

import cx from './MainOffers.scss';

/* Component Types */
interface IMainOffersProps
    extends IWithClassName,
        IWithDeviceType,
        IWithQaAttributes {
    offers: (IHotelOffer | HotelRoomOfferType)[];
    nightsCount?: number;
    partnerOffers?: IHotelPartnerOffer[];
    operatorById?: OperatorByIdType;
    isRoomGroup: boolean;
    isFullOffersTab?: boolean;
    onOfferSelect?: () => void;
    onOfferWatchButtonClick?: (roomId: string) => void;
    chosenId?: string;
}

const MainOffers: FunctionComponent<IMainOffersProps> = ({
    onOfferSelect,
    offers,
    partnerOffers,
    operatorById,
    isFullOffersTab,
    isRoomGroup,
    nightsCount,
    onOfferWatchButtonClick,
    className,
    deviceType,
    chosenId,
    ...rest
}) => {
    const location = useLocation();
    const queryByLocation = getQueryByLocation(location);
    const {isHeadless} = getViewModePageParamsByQuery(queryByLocation);
    const {roomsMatching, hotelsPercentDiscount} =
        useSelector(experimentsSelector);
    const {isMobile} = deviceType;

    const handleOfferSelect = useCallback((): void => {
        onOfferSelect?.();
    }, [onOfferSelect]);

    /* Helper */

    const visibleOffers = useMemo(():
        | (IHotelOffer | HotelRoomOfferType)[]
        | undefined => {
        if (offers) {
            if (
                isFullOffersTab ||
                offers.length <=
                    BOY_META_MAIN_TAB_MAX_OFFERS + BOY_META_MAIN_TAB_MAX_LIMITS
            ) {
                return offers;
            }

            return offers.slice(0, BOY_META_MAIN_TAB_MAX_OFFERS);
        }

        return undefined;
    }, [offers, isFullOffersTab]);

    const groupedOffers = useMemo(() => {
        const grouped = groupBy(
            visibleOffers as IHotelOffer[],
            v =>
                `${v.roomId}${v.cancellationInfo?.refundType || ''}${
                    v.mealType?.id || ''
                }`,
        );

        return Object.entries(grouped).map(([roomId, roomOffers]) => {
            return {
                roomId,
                firstOffer: roomOffers[0],
                roomOffers,
            };
        });
    }, [visibleOffers]);

    if (!visibleOffers || !operatorById) {
        return null;
    }

    if (isRoomGroup && roomsMatching) {
        return (
            <div
                className={cx(
                    className,
                    'root',
                    deviceMods('root', deviceType),
                )}
                {...prepareQaAttributes(rest)}
            >
                <Flex
                    className={cx('matchedOffersTitle')}
                    alignItems="baseline"
                >
                    <Text weight="bold" size={isMobile ? 'l' : 'xl'}>
                        {i18nBlock.tariffs()}
                    </Text>
                    {nightsCount && (
                        <Text size="m">
                            {i18nBlock.priceForNights({nights: nightsCount})}
                        </Text>
                    )}
                </Flex>
                {groupedOffers.map(
                    ({roomId, roomOffers, firstOffer}, index): ReactNode => (
                        <MatchedOffers
                            key={roomId}
                            mainOffer={firstOffer}
                            offers={roomOffers}
                            operatorById={operatorById}
                            isLast={groupedOffers.length - 1 === index}
                            hotelsPercentDiscount={hotelsPercentDiscount}
                            {...prepareQaAttributes({
                                parent: rest,
                                current: 'matchedOffer',
                                key: roomId,
                            })}
                        />
                    ),
                )}
            </div>
        );
    }

    return (
        <div
            className={cx(className, 'root', deviceMods('root', deviceType), {
                root_noRooms: !isRoomGroup,
            })}
            {...prepareQaAttributes({
                parent: rest,
                current: 'offersList',
            })}
        >
            {visibleOffers.map(
                (
                    offer: IHotelOffer | HotelRoomOfferType,
                    index: number,
                ): ReactNode => (
                    <YandexOffer
                        isHeadless={isHeadless}
                        isNameHidden={isRoomGroup}
                        key={offer.id}
                        onOfferSelect={handleOfferSelect}
                        chosenId={
                            isFullOffersTab && !isRoomGroup
                                ? chosenId
                                : undefined
                        }
                        onOfferWatchButtonClick={onOfferWatchButtonClick}
                        offer={offer as IHotelOffer}
                        nightsCount={nightsCount}
                        isFullOffersTab={isFullOffersTab}
                        hotelsPercentDiscount={hotelsPercentDiscount}
                        {...prepareQaAttributes({
                            parent: rest,
                            current: 'mainOffer',
                            key: index,
                        })}
                    />
                ),
            )}
        </div>
    );
};

export default MainOffers;
