import React, {useCallback, useEffect, useRef} from 'react';

import {IHotelOffer} from 'types/hotels/offer/IHotelOffer';
import {IWithClassName} from 'types/withClassName';

import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {deviceMods} from 'utilities/stylesUtils';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import scrollToNode from 'utilities/dom/scrollToNode';

import * as i18nOffer from 'i18n/hotels-HotelPageMainOffersInfo';
import * as i18nRoom from 'i18n/hotels-HotelPageRooms';

/* Components */
import HotelOfferLabels from 'projects/hotels/components/HotelOfferLabels/HotelOfferLabels';
import Offer from 'projects/hotels/pages/HotelPage/components/OffersInfo/components/Offer/Offer';

import cx from './YandexOffer.scss';

export interface IYandexOfferProps extends IWithClassName, IWithQaAttributes {
    offer: IHotelOffer;
    isNameHidden: boolean;
    onOfferSelect: () => void;
    nightsCount?: number;
    isHeadless?: boolean;
    isFullOffersTab?: boolean;
    hotelsPercentDiscount?: boolean;
    onOfferWatchButtonClick?: (roomId: string) => void;
    chosenId?: string;
}

const YandexOffer: React.FC<IYandexOfferProps> = props => {
    const {
        className,
        offer,
        isHeadless,
        isNameHidden,
        onOfferSelect,
        nightsCount,
        isFullOffersTab,
        onOfferWatchButtonClick,
        chosenId,
        hotelsPercentDiscount,
    } = props;
    const deviceType = useDeviceType();
    const {price, name, landingUrl, discountInfo, offerYandexPlusInfo, badges} =
        offer;

    const hasNoRoom = !isNameHidden && !isFullOffersTab;
    const offerRef = useRef<HTMLDivElement>(null);

    const handleWatchButtonClick = useCallback(() => {
        onOfferWatchButtonClick?.(offer.id);
    }, [onOfferWatchButtonClick, offer]);

    useEffect(() => {
        if (isFullOffersTab && chosenId === offer.id && offerRef.current) {
            scrollToNode(offerRef.current, {behavior: 'smooth'});
        }
    }, [isFullOffersTab, chosenId, offer, offerRef]);

    return (
        <div
            className={cx(className, 'root', deviceMods('root', deviceType), {
                root_noRoom: !isNameHidden && !isFullOffersTab,
                root_noRoom_offers: !isNameHidden && isFullOffersTab,
            })}
            ref={offerRef}
            {...prepareQaAttributes(props)}
        >
            <Offer
                className={cx('offer')}
                title={isNameHidden ? undefined : name}
                isWide
                type={
                    deviceType.isDesktop && !hasNoRoom
                        ? 'horizontal'
                        : 'compact'
                }
                labels={
                    <div className={cx('labels')}>
                        <HotelOfferLabels
                            offer={offer}
                            cancellationLabelClassName={cx(
                                'label',
                                'cancellation',
                            )}
                            cancellationTextWithIconClassName={cx(
                                'cancellation_textWithIcon',
                            )}
                            mealLabelClassName={cx('label', 'meal')}
                            isMealIconHidden={true}
                            {...prepareQaAttributes({
                                parent: props,
                                current: 'hotelOfferLabels',
                            })}
                        />
                    </div>
                }
                badges={badges}
                price={price}
                plusInfo={offerYandexPlusInfo}
                discountInfo={discountInfo}
                hotelsPercentDiscount={hotelsPercentDiscount}
                justifyContent={deviceType.isMobile ? 'flex-start' : 'flex-end'}
                buttonTheme="primary"
                buttonText={
                    isFullOffersTab
                        ? i18nOffer.selectAction()
                        : i18nRoom.watch()
                }
                buttonUrl={isFullOffersTab ? landingUrl : undefined}
                nightsCount={nightsCount}
                onOfferSelect={
                    isFullOffersTab ? onOfferSelect : handleWatchButtonClick
                }
                isHeadless={isHeadless}
                showNightsCount={Boolean(nightsCount)}
                {...prepareQaAttributes(props)}
            />
        </div>
    );
};

export default YandexOffer;
