import React, {ReactNode, useMemo} from 'react';

import {MIN_PERCENT_TO_SHOW} from 'projects/hotels/constants/discount';

import {
    IDiscountInfo,
    EHotelBadgeType,
    IHotelOrOfferBadge,
    IOfferYandexPlusInfo,
} from 'types/hotels/offer/IHotelOffer';
import {IWithClassName} from 'types/withClassName';
import {TLinkTarget} from 'types/common/TLinkTarget';

import IPrice from 'utilities/currency/PriceInterface';
import {getOperatorLinkPropsByParams} from 'projects/hotels/utilities/getOperatorLinkPropsByParams/getOperatorLinkPropsByParams';
import {
    getQa,
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {deviceModDesktop, deviceMods} from 'utilities/stylesUtils';
import {useWhiteLabelConfig} from 'utilities/hooks/useWhiteLabelConfig';
import getWhiteLabelBadge from 'components/WhiteLabelLoyaltyBadge/utilities/getWhiteLabelBadge';
import {getYandexPlusBadge} from 'utilities/hotelOffer/getYandexPlusBadge';

import * as i18nBlock from 'i18n/hotels-HotelPageOffersInfo';

import WhiteLabelLoyaltyBadge from 'components/WhiteLabelLoyaltyBadge/WhiteLabelLoyaltyBadge';
import Text from 'components/Text/Text';
import Price from 'components/Price/Price';
import ButtonLink from 'components/ButtonLink/ButtonLink';
import HotelOfferBadges from 'projects/hotels/components/HotelOfferBadges/HotelOfferBadges';
import Flex, {IFlexProps} from 'components/Flex/Flex';
import PlusInfoPopup from 'components/PlusInfoPopup/PlusInfoPopup';
import YandexEdaPromo from 'projects/hotels/components/YandexEdaPromo/YandexEdaPromo';
import DiscountInfo from 'projects/hotels/components/HotelCard/components/DiscountInfo/DiscountInfo';

import cx from './Offer.scss';

export interface IOfferCollapsed
    extends IWithClassName,
        IWithQaAttributes,
        Pick<IFlexProps, 'justifyContent'> {
    labels: React.ReactElement;
    price: IPrice;
    discountInfo?: IDiscountInfo;
    hotelsPercentDiscount?: boolean;
    buttonTheme: 'secondary' | 'primary';
    buttonText: string;
    showNightsCount: boolean;
    nightsCount?: number;
    title?: React.ReactElement | string;
    onOfferSelect?: React.MouseEventHandler;
    buttonUrl?: string;
    isHeadless?: boolean;
    badges?: IHotelOrOfferBadge[];
    type?: 'compact' | 'horizontal';
    isWide?: boolean;
    isPartner?: boolean;
    plusInfo?: IOfferYandexPlusInfo;
    linkTarget?: TLinkTarget;
}

const Offer: React.FC<IOfferCollapsed> = props => {
    const {
        className,
        title,
        labels,
        isHeadless,
        onOfferSelect,
        price,
        discountInfo,
        justifyContent = 'flex-start',
        buttonTheme,
        buttonUrl,
        buttonText,
        nightsCount,
        badges,
        type = 'compact',
        isWide,
        showNightsCount,
        isPartner,
        plusInfo,
        linkTarget,
        hotelsPercentDiscount,
    } = props;
    const deviceType = useDeviceType();
    const {isWhiteLabel} = useWhiteLabelConfig();
    const {isMobile, isDesktop} = deviceType;
    const yandexEdaBadge = badges?.find(
        badge => badge.id === EHotelBadgeType.YANDEX_EDA,
    );
    const showDiscountPercent =
        hotelsPercentDiscount &&
        discountInfo?.percent &&
        discountInfo.percent >= MIN_PERCENT_TO_SHOW;

    const propsQa = getQa(props);

    const renderBadges = (): ReactNode => {
        if (!badges?.length) {
            return null;
        }

        return (
            <HotelOfferBadges
                badges={badges}
                above={type === 'horizontal' ? 1 : 4}
                below={2}
                alignItems={isMobile ? 'flex-start' : 'flex-end'}
            />
        );
    };

    const whiteLabelBadge = useMemo(() => {
        if (!badges?.length) {
            return null;
        }

        const badge = getWhiteLabelBadge(badges);

        if (badge === undefined) {
            return null;
        }

        return <WhiteLabelLoyaltyBadge badge={badge} />;
    }, [badges]);

    const plusBadge = getYandexPlusBadge(badges);

    const plusInfoBlock = useMemo(() => {
        return (
            plusInfo?.points && (
                <PlusInfoPopup
                    className={cx(
                        'plusOffer',
                        deviceModDesktop('plusOffer', deviceType),
                    )}
                    title={plusBadge?.additionalPromoInfo?.title}
                    description={plusBadge?.additionalPromoInfo?.text}
                    linkUrl={plusBadge?.additionalPromoInfo?.link?.url}
                    linkText={plusBadge?.additionalPromoInfo?.link?.text}
                    buttonText={plusBadge?.text}
                    badgeId={plusBadge?.id}
                    {...plusInfo}
                    {...prepareQaAttributes({
                        parent: propsQa,
                        current: 'plusInfo',
                    })}
                />
            )
        );
    }, [deviceType, plusInfo, propsQa]);

    const bonusBlock = isWhiteLabel ? whiteLabelBadge : plusInfoBlock;

    return (
        <section
            className={cx(className, deviceMods('root', deviceType), {
                root_horizontal: type === 'horizontal',
                root_wide: isWide,
            })}
            {...prepareQaAttributes(propsQa)}
        >
            {title && (
                <div
                    className={cx('offerName')}
                    {...prepareQaAttributes({
                        parent: propsQa,
                        current: 'offerName',
                    })}
                >
                    {title}
                </div>
            )}
            <div
                className={cx('labels')}
                {...prepareQaAttributes({
                    parent: propsQa,
                    current: 'labels',
                })}
            >
                {labels}
            </div>
            {type !== 'horizontal' && renderBadges()}
            <div
                className={cx('price', {
                    price_short: !showNightsCount,
                })}
            >
                <div className={cx('priceNights')}>
                    <Flex
                        alignItems="center"
                        justifyContent={justifyContent}
                        flexWrap="wrap"
                        between="2"
                        inline
                    >
                        <Price
                            className={cx('price__value')}
                            {...price}
                            color={
                                discountInfo?.strikethroughPrice &&
                                !showDiscountPercent
                                    ? 'alert'
                                    : undefined
                            }
                            {...prepareQaAttributes({
                                parent: propsQa,
                                current: 'price',
                            })}
                        />
                        {discountInfo && (
                            <DiscountInfo
                                discountInfo={discountInfo}
                                hotelsPercentDiscount={hotelsPercentDiscount}
                                percentSize={deviceType.isDesktop ? 's' : 'm'}
                            />
                        )}
                    </Flex>
                    {showNightsCount && nightsCount && (
                        <Text
                            size="s-inset"
                            className={cx('price__nights')}
                            {...prepareQaAttributes({
                                parent: props,
                                current: 'nightsCount',
                            })}
                        >
                            {i18nBlock.nightsCount({nightsCount})}
                        </Text>
                    )}
                    {isDesktop && bonusBlock}
                    {isDesktop && yandexEdaBadge && (
                        <YandexEdaPromo badge={yandexEdaBadge} />
                    )}
                    {type === 'horizontal' && renderBadges()}
                </div>
                <ButtonLink
                    theme={buttonTheme}
                    size={isMobile || (!isPartner && !isWide) ? 'l' : 'm-inset'}
                    url={buttonUrl}
                    onClick={onOfferSelect}
                    target={linkTarget}
                    {...getOperatorLinkPropsByParams({isHeadless})}
                    {...prepareQaAttributes({
                        parent: propsQa,
                        current: 'bookButton',
                    })}
                >
                    {buttonText}
                </ButtonLink>
            </div>
            {isMobile && bonusBlock}
            {isMobile && yandexEdaBadge && (
                <div>
                    <YandexEdaPromo badge={yandexEdaBadge} />
                </div>
            )}
        </section>
    );
};

export default Offer;
